#!/bin/bash

# GitHub Classroom Assignment Creator
# This script creates a new assignment in a GitHub Classroom using the GitHub CLI

set -e  # Exit on any error

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Function to show usage
show_usage() {
    echo "Usage: $0 [OPTIONS]"
    echo ""
    echo "Creates a GitHub Classroom assignment with specified parameters."
    echo ""
    echo "Options:"
    echo "  -c, --classroom CLASSROOM_ID    GitHub Classroom ID (required)"
    echo "  -t, --title TITLE              Assignment title (required)"
    echo "  -s, --starter-repo REPO        Starter code repository (org/repo format)"
    echo "  -d, --deadline DEADLINE        Assignment deadline (YYYY-MM-DD format)"
    echo "  -p, --prefix PREFIX            Repository name prefix (default: assignment)"
    echo "  --public                       Make student repositories public (default: private)"
    echo "  --individual                   Individual assignment (default: group)"
    echo "  --max-teams MAX                Maximum number of teams (for group assignments)"
    echo "  --max-members MAX              Maximum members per team (for group assignments)"
    echo "  -h, --help                     Show this help message"
    echo ""
    echo "Examples:"
    echo "  $0 -c 12345 -t \"Data Pipeline Project\" -s ZipCodeCore/CentralLibraryData"
    echo "  $0 -c 12345 -t \"Individual Exercise\" -s myorg/starter-repo --individual"
    echo ""
    echo "Note: This script requires the GitHub CLI (gh) to be installed and authenticated."
}

# Default values
CLASSROOM_ID=""
ASSIGNMENT_TITLE=""
STARTER_REPO=""
DEADLINE=""
REPO_PREFIX="assignment"
IS_PUBLIC="false"
IS_INDIVIDUAL="false"
MAX_TEAMS=""
MAX_MEMBERS=""

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        -c|--classroom)
            CLASSROOM_ID="$2"
            shift 2
            ;;
        -t|--title)
            ASSIGNMENT_TITLE="$2"
            shift 2
            ;;
        -s|--starter-repo)
            STARTER_REPO="$2"
            shift 2
            ;;
        -d|--deadline)
            DEADLINE="$2"
            shift 2
            ;;
        -p|--prefix)
            REPO_PREFIX="$2"
            shift 2
            ;;
        --public)
            IS_PUBLIC="true"
            shift
            ;;
        --individual)
            IS_INDIVIDUAL="true"
            shift
            ;;
        --max-teams)
            MAX_TEAMS="$2"
            shift 2
            ;;
        --max-members)
            MAX_MEMBERS="$2"
            shift 2
            ;;
        -h|--help)
            show_usage
            exit 0
            ;;
        *)
            print_error "Unknown option: $1"
            show_usage
            exit 1
            ;;
    esac
done

# Validate required parameters
if [[ -z "$CLASSROOM_ID" ]]; then
    print_error "Classroom ID is required. Use -c or --classroom option."
    show_usage
    exit 1
fi

if [[ -z "$ASSIGNMENT_TITLE" ]]; then
    print_error "Assignment title is required. Use -t or --title option."
    show_usage
    exit 1
fi

# Check if GitHub CLI is installed
if ! command -v gh &> /dev/null; then
    print_error "GitHub CLI (gh) is not installed. Please install it first:"
    echo "  brew install gh"
    echo "  or visit: https://cli.github.com/"
    exit 1
fi

# Check if user is authenticated with GitHub CLI
if ! gh auth status &> /dev/null; then
    print_error "Not authenticated with GitHub CLI. Please run:"
    echo "  gh auth login"
    exit 1
fi

# Validate starter repo format if provided
if [[ -n "$STARTER_REPO" ]]; then
    if [[ ! "$STARTER_REPO" =~ ^[^/]+/[^/]+$ ]]; then
        print_error "Starter repository must be in format 'org/repo'"
        exit 1
    fi
fi

# Validate deadline format if provided
if [[ -n "$DEADLINE" ]]; then
    if [[ ! "$DEADLINE" =~ ^[0-9]{4}-[0-9]{2}-[0-9]{2}$ ]]; then
        print_error "Deadline must be in YYYY-MM-DD format"
        exit 1
    fi
fi

print_info "Creating GitHub Classroom assignment..."
print_info "Classroom ID: $CLASSROOM_ID"
print_info "Title: $ASSIGNMENT_TITLE"
print_info "Starter repo: ${STARTER_REPO:-"None"}"
print_info "Assignment type: $([ "$IS_INDIVIDUAL" = "true" ] && echo "Individual" || echo "Group")"

# Build the GitHub CLI command
GH_CMD="gh classroom assignment create"
GH_CMD="$GH_CMD --classroom $CLASSROOM_ID"
GH_CMD="$GH_CMD --title \"$ASSIGNMENT_TITLE\""
GH_CMD="$GH_CMD --prefix \"$REPO_PREFIX\""

if [[ -n "$STARTER_REPO" ]]; then
    GH_CMD="$GH_CMD --starter-code-repository $STARTER_REPO"
fi

if [[ -n "$DEADLINE" ]]; then
    GH_CMD="$GH_CMD --deadline $DEADLINE"
fi

if [[ "$IS_PUBLIC" = "true" ]]; then
    GH_CMD="$GH_CMD --public"
fi

if [[ "$IS_INDIVIDUAL" = "true" ]]; then
    GH_CMD="$GH_CMD --individual"
else
    # Group assignment options
    if [[ -n "$MAX_TEAMS" ]]; then
        GH_CMD="$GH_CMD --max-teams $MAX_TEAMS"
    fi
    if [[ -n "$MAX_MEMBERS" ]]; then
        GH_CMD="$GH_CMD --max-members $MAX_MEMBERS"
    fi
fi

# Execute the command
print_info "Executing: $GH_CMD"
echo ""

if eval "$GH_CMD"; then
    print_success "Assignment created successfully!"
    echo ""
    print_info "Students can access the assignment through your GitHub Classroom."
    print_info "Assignment repositories will be created with prefix: $REPO_PREFIX"
else
    print_error "Failed to create assignment. Please check the error messages above."
    exit 1
fi

# Additional helpful information
echo ""
print_info "Next steps:"
echo "1. Share the assignment link with your students"
echo "2. Monitor student progress in your GitHub Classroom dashboard"
echo "3. Students will get repositories named: $REPO_PREFIX-{student-name}"

if [[ -n "$STARTER_REPO" ]]; then
    echo "4. Student repositories will be initialized with code from: $STARTER_REPO"
fi
