/** 

This Java implementation showcases the core principles of double-entry accounting, using ducats as 
the currency in honor of the Medici banking dynasty. Here's what the code demonstrates:

1. **The Fundamental Principle**: Every transaction affects at least two accounts 
(the double-entry principle), and the sum of debits must always equal the sum of credits.

2. **Five Main Account Types**:
   - Assets: Resources owned by the business
   - Liabilities: Debts owed by the business
   - Equity: Owner's interest in the business
   - Revenue: Income earned by the business
   - Expenses: Costs incurred by the business

3. **Account Balance Rules**:
   - Assets and Expenses: Increased by debits, decreased by credits
   - Liabilities, Equity, and Revenue: Increased by credits, decreased by debits

4. **Key Financial Reports**:
   - Trial Balance: Verifies that total debits equal total credits
   - Balance Sheet: Shows Assets = Liabilities + Equity
   - Income Statement: Shows Revenue - Expenses = Net Income

The example simulates transactions for the Medici Bank in the year 1397, 
including initial capitalization, loans with interest (a key banking activity), 
property acquisition, and operating expenses.

*/
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.*;

/**
 * A double-entry accounting system implementation inspired by the 
 * Florentine banking practices of the deMedici family.
 */
public class MediciBankingSystem {

    public static void main(String[] args) {
        // Create a new ledger for our banking operations
        Ledger mediciLedger = new Ledger("Medici Family Bank");
        
        // Define our chart of accounts
        Account cash = mediciLedger.createAccount("Cash", AccountType.ASSET);
        Account accounts_receivable = mediciLedger.createAccount("Accounts Receivable", AccountType.ASSET);
        Account inventory = mediciLedger.createAccount("Inventory", AccountType.ASSET);
        Account land = mediciLedger.createAccount("Land", AccountType.ASSET);
        
        Account accounts_payable = mediciLedger.createAccount("Accounts Payable", AccountType.LIABILITY);
        Account loans = mediciLedger.createAccount("Loans", AccountType.LIABILITY);
        
        Account capital = mediciLedger.createAccount("Owner's Capital", AccountType.EQUITY);
        Account retained_earnings = mediciLedger.createAccount("Retained Earnings", AccountType.EQUITY);
        
        Account revenue = mediciLedger.createAccount("Revenue", AccountType.REVENUE);
        Account interest_income = mediciLedger.createAccount("Interest Income", AccountType.REVENUE);
        
        Account expenses = mediciLedger.createAccount("Expenses", AccountType.EXPENSE);
        Account wages = mediciLedger.createAccount("Wages", AccountType.EXPENSE);
        
        // Starting our banking operations with initial capital
        System.out.println("=== STARTING THE MEDICI BANK ===");
        mediciLedger.recordTransaction(
            LocalDate.of(1397, 1, 1),
            "Initial investment from Giovanni de' Medici",
            new TransactionEntry(cash, new BigDecimal("10000.00")),
            new TransactionEntry(capital, new BigDecimal("10000.00"))
        );
        
        // Recording a loan to a wool merchant
        mediciLedger.recordTransaction(
            LocalDate.of(1397, 2, 15),
            "Loan to Wool Merchant",
            new TransactionEntry(accounts_receivable, new BigDecimal("2000.00")),
            new TransactionEntry(cash, new BigDecimal("2000.00"))
        );
        
        // Receiving partial payment with interest
        mediciLedger.recordTransaction(
            LocalDate.of(1397, 8, 10),
            "Partial loan repayment from Wool Merchant with interest",
            new TransactionEntry(cash, new BigDecimal("1200.00")),
            new TransactionEntry(accounts_receivable, new BigDecimal("1000.00")),
            new TransactionEntry(interest_income, new BigDecimal("200.00"))
        );
        
        // Purchasing land for a new banking house
        mediciLedger.recordTransaction(
            LocalDate.of(1397, 9, 5),
            "Purchase of land for new Medici banking house",
            new TransactionEntry(land, new BigDecimal("3000.00")),
            new TransactionEntry(cash, new BigDecimal("3000.00"))
        );
        
        // Paying wages to bank employees
        mediciLedger.recordTransaction(
            LocalDate.of(1397, 12, 1),
            "Quarterly wages for bank employees",
            new TransactionEntry(wages, new BigDecimal("800.00")),
            new TransactionEntry(cash, new BigDecimal("800.00"))
        );
        
        // Print the trial balance to verify our accounting is balanced
        System.out.println("\n=== MEDICI BANK TRIAL BALANCE (Year 1397) ===");
        mediciLedger.printTrialBalance();
        
        // Print the balance sheet
        System.out.println("\n=== MEDICI BANK BALANCE SHEET (Year 1397) ===");
        mediciLedger.printBalanceSheet();
        
        // Print the income statement
        System.out.println("\n=== MEDICI BANK INCOME STATEMENT (Year 1397) ===");
        mediciLedger.printIncomeStatement();
    }
}

/**
 * Represents a financial account in the double-entry system
 */
class Account {
    private String name;
    private AccountType type;
    private BigDecimal balance;
    
    public Account(String name, AccountType type) {
        this.name = name;
        this.type = type;
        this.balance = BigDecimal.ZERO;
    }
    
    public String getName() {
        return name;
    }
    
    public AccountType getType() {
        return type;
    }
    
    public BigDecimal getBalance() {
        return balance;
    }
    
    /**
     * Debits increase ASSET and EXPENSE accounts
     * Debits decrease LIABILITY, EQUITY, and REVENUE accounts
     */
    public void debit(BigDecimal amount) {
        if (type == AccountType.ASSET || type == AccountType.EXPENSE) {
            balance = balance.add(amount);
        } else {
            balance = balance.subtract(amount);
        }
    }
    
    /**
     * Credits decrease ASSET and EXPENSE accounts
     * Credits increase LIABILITY, EQUITY, and REVENUE accounts
     */
    public void credit(BigDecimal amount) {
        if (type == AccountType.ASSET || type == AccountType.EXPENSE) {
            balance = balance.subtract(amount);
        } else {
            balance = balance.add(amount);
        }
    }
    
    @Override
    public String toString() {
        return name + " (" + type + "): " + balance + " ducats";
    }
}

/**
 * The different types of accounts in double-entry accounting
 */
enum AccountType {
    ASSET,      // Resources owned by the business
    LIABILITY,  // Debts owed by the business
    EQUITY,     // Owner's interest in the business
    REVENUE,    // Income earned by the business
    EXPENSE     // Costs incurred by the business
}

/**
 * Represents a single entry in a transaction
 */
class TransactionEntry {
    private Account account;
    private BigDecimal amount;
    
    public TransactionEntry(Account account, BigDecimal amount) {
        this.account = account;
        this.amount = amount;
    }
    
    public Account getAccount() {
        return account;
    }
    
    public BigDecimal getAmount() {
        return amount;
    }
}

/**
 * Represents a complete financial transaction in the double-entry system
 */
class Transaction {
    private String description;
    private LocalDate date;
    private List<TransactionEntry> debits;
    private List<TransactionEntry> credits;
    
    public Transaction(LocalDate date, String description) {
        this.date = date;
        this.description = description;
        this.debits = new ArrayList<>();
        this.credits = new ArrayList<>();
    }
    
    public void addDebit(TransactionEntry entry) {
        debits.add(entry);
    }
    
    public void addCredit(TransactionEntry entry) {
        credits.add(entry);
    }
    
    public boolean isBalanced() {
        BigDecimal totalDebits = debits.stream()
            .map(TransactionEntry::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
            
        BigDecimal totalCredits = credits.stream()
            .map(TransactionEntry::getAmount)
            .reduce(BigDecimal.ZERO, BigDecimal::add);
            
        return totalDebits.compareTo(totalCredits) == 0;
    }
    
    public void post() {
        // Apply all debits
        for (TransactionEntry entry : debits) {
            entry.getAccount().debit(entry.getAmount());
        }
        
        // Apply all credits
        for (TransactionEntry entry : credits) {
            entry.getAccount().credit(entry.getAmount());
        }
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Transaction: ").append(date).append(" - ").append(description).append("\n");
        
        sb.append("  Debits:\n");
        for (TransactionEntry entry : debits) {
            sb.append("    ").append(entry.getAccount().getName())
              .append(": ").append(entry.getAmount()).append(" ducats\n");
        }
        
        sb.append("  Credits:\n");
        for (TransactionEntry entry : credits) {
            sb.append("    ").append(entry.getAccount().getName())
              .append(": ").append(entry.getAmount()).append(" ducats\n");
        }
        
        return sb.toString();
    }
}

/**
 * The main ledger that keeps track of all accounts and transactions
 */
class Ledger {
    private String name;
    private List<Account> accounts;
    private List<Transaction> transactions;
    
    public Ledger(String name) {
        this.name = name;
        this.accounts = new ArrayList<>();
        this.transactions = new ArrayList<>();
    }
    
    public Account createAccount(String name, AccountType type) {
        Account account = new Account(name, type);
        accounts.add(account);
        return account;
    }
    
    /**
     * Records a transaction with any number of debits and credits,
     * ensuring that debits = credits (the fundamental principle of double-entry)
     */
    public void recordTransaction(LocalDate date, String description, 
                                  TransactionEntry... entries) {
        Transaction transaction = new Transaction(date, description);
        
        // Separate entries into debits and credits based on account type
        for (TransactionEntry entry : entries) {
            Account account = entry.getAccount();
            AccountType type = account.getType();
            
            // For asset and expense accounts, positive amounts are debits
            // For liability, equity, and revenue accounts, positive amounts are credits
            if (type == AccountType.ASSET || type == AccountType.EXPENSE) {
                if (entry.getAmount().compareTo(BigDecimal.ZERO) >= 0) {
                    transaction.addDebit(entry);
                } else {
                    // Negative amount means we're crediting the account
                    transaction.addCredit(new TransactionEntry(
                        account, entry.getAmount().abs()));
                }
            } else {
                if (entry.getAmount().compareTo(BigDecimal.ZERO) >= 0) {
                    transaction.addCredit(entry);
                } else {
                    // Negative amount means we're debiting the account
                    transaction.addDebit(new TransactionEntry(
                        account, entry.getAmount().abs()));
                }
            }
        }
        
        // Verify that the transaction is balanced
        if (!transaction.isBalanced()) {
            throw new IllegalArgumentException("Transaction is not balanced: debits must equal credits");
        }
        
        // Post the transaction to update account balances
        transaction.post();
        
        // Record the transaction in the ledger
        transactions.add(transaction);
        System.out.println(transaction);
    }
    
    /**
     * Prints a trial balance to verify that debits = credits across all accounts
     */
    public void printTrialBalance() {
        BigDecimal totalDebits = BigDecimal.ZERO;
        BigDecimal totalCredits = BigDecimal.ZERO;
        
        System.out.printf("%-30s %-15s %-15s%n", "Account", "Debit (Ducats)", "Credit (Ducats)");
        System.out.println("-".repeat(60));
        
        for (Account account : accounts) {
            BigDecimal balance = account.getBalance();
            
            // For the trial balance, we show positive balances in their normal position
            // Assets and Expenses with positive balances are shown as debits
            // Liabilities, Equity, and Revenue with positive balances are shown as credits
            if (account.getType() == AccountType.ASSET || 
                account.getType() == AccountType.EXPENSE) {
                if (balance.compareTo(BigDecimal.ZERO) > 0) {
                    System.out.printf("%-30s %-15s %-15s%n", 
                        account.getName(), balance.setScale(2, RoundingMode.HALF_UP), "");
                    totalDebits = totalDebits.add(balance);
                } else if (balance.compareTo(BigDecimal.ZERO) < 0) {
                    System.out.printf("%-30s %-15s %-15s%n", 
                        account.getName(), "", balance.abs().setScale(2, RoundingMode.HALF_UP));
                    totalCredits = totalCredits.add(balance.abs());
                }
            } else {
                if (balance.compareTo(BigDecimal.ZERO) > 0) {
                    System.out.printf("%-30s %-15s %-15s%n", 
                        account.getName(), "", balance.setScale(2, RoundingMode.HALF_UP));
                    totalCredits = totalCredits.add(balance);
                } else if (balance.compareTo(BigDecimal.ZERO) < 0) {
                    System.out.printf("%-30s %-15s %-15s%n", 
                        account.getName(), balance.abs().setScale(2, RoundingMode.HALF_UP), "");
                    totalDebits = totalDebits.add(balance.abs());
                }
            }
        }
        
        System.out.println("-".repeat(60));
        System.out.printf("%-30s %-15s %-15s%n", 
            "TOTAL", totalDebits.setScale(2, RoundingMode.HALF_UP), 
            totalCredits.setScale(2, RoundingMode.HALF_UP));
            
        if (totalDebits.compareTo(totalCredits) == 0) {
            System.out.println("\nThe books are balanced! ✓");
        } else {
            System.out.println("\nWARNING: The books are NOT balanced! ✗");
        }
    }
    
    /**
     * Prints a balance sheet (Assets = Liabilities + Equity)
     */
    public void printBalanceSheet() {
        BigDecimal totalAssets = BigDecimal.ZERO;
        BigDecimal totalLiabilities = BigDecimal.ZERO;
        BigDecimal totalEquity = BigDecimal.ZERO;
        
        // Print Assets
        System.out.println("ASSETS");
        System.out.println("-".repeat(40));
        for (Account account : accounts) {
            if (account.getType() == AccountType.ASSET && account.getBalance().compareTo(BigDecimal.ZERO) != 0) {
                System.out.printf("%-30s %10s%n", 
                    account.getName(), account.getBalance().setScale(2, RoundingMode.HALF_UP));
                totalAssets = totalAssets.add(account.getBalance());
            }
        }
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "TOTAL ASSETS", totalAssets.setScale(2, RoundingMode.HALF_UP));
        System.out.println();
        
        // Print Liabilities
        System.out.println("LIABILITIES");
        System.out.println("-".repeat(40));
        for (Account account : accounts) {
            if (account.getType() == AccountType.LIABILITY && account.getBalance().compareTo(BigDecimal.ZERO) != 0) {
                System.out.printf("%-30s %10s%n", 
                    account.getName(), account.getBalance().setScale(2, RoundingMode.HALF_UP));
                totalLiabilities = totalLiabilities.add(account.getBalance());
            }
        }
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "TOTAL LIABILITIES", totalLiabilities.setScale(2, RoundingMode.HALF_UP));
        System.out.println();
        
        // Print Equity
        System.out.println("EQUITY");
        System.out.println("-".repeat(40));
        for (Account account : accounts) {
            if (account.getType() == AccountType.EQUITY && account.getBalance().compareTo(BigDecimal.ZERO) != 0) {
                System.out.printf("%-30s %10s%n", 
                    account.getName(), account.getBalance().setScale(2, RoundingMode.HALF_UP));
                totalEquity = totalEquity.add(account.getBalance());
            }
        }
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "TOTAL EQUITY", totalEquity.setScale(2, RoundingMode.HALF_UP));
        System.out.println();
        
        // Verify the accounting equation: Assets = Liabilities + Equity
        System.out.println("ACCOUNTING EQUATION");
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "Total Assets", totalAssets.setScale(2, RoundingMode.HALF_UP));
        System.out.printf("%-30s %10s%n", "Total Liabilities + Equity", 
            totalLiabilities.add(totalEquity).setScale(2, RoundingMode.HALF_UP));
            
        if (totalAssets.compareTo(totalLiabilities.add(totalEquity)) == 0) {
            System.out.println("\nThe accounting equation is balanced! ✓");
        } else {
            System.out.println("\nWARNING: The accounting equation is NOT balanced! ✗");
        }
    }
    
    /**
     * Prints an income statement (Revenue - Expenses = Net Income)
     */
    public void printIncomeStatement() {
        BigDecimal totalRevenue = BigDecimal.ZERO;
        BigDecimal totalExpenses = BigDecimal.ZERO;
        
        // Print Revenue
        System.out.println("REVENUE");
        System.out.println("-".repeat(40));
        for (Account account : accounts) {
            if (account.getType() == AccountType.REVENUE && account.getBalance().compareTo(BigDecimal.ZERO) != 0) {
                System.out.printf("%-30s %10s%n", 
                    account.getName(), account.getBalance().setScale(2, RoundingMode.HALF_UP));
                totalRevenue = totalRevenue.add(account.getBalance());
            }
        }
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "TOTAL REVENUE", totalRevenue.setScale(2, RoundingMode.HALF_UP));
        System.out.println();
        
        // Print Expenses
        System.out.println("EXPENSES");
        System.out.println("-".repeat(40));
        for (Account account : accounts) {
            if (account.getType() == AccountType.EXPENSE && account.getBalance().compareTo(BigDecimal.ZERO) != 0) {
                System.out.printf("%-30s %10s%n", 
                    account.getName(), account.getBalance().setScale(2, RoundingMode.HALF_UP));
                totalExpenses = totalExpenses.add(account.getBalance());
            }
        }
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "TOTAL EXPENSES", totalExpenses.setScale(2, RoundingMode.HALF_UP));
        System.out.println();
        
        // Calculate Net Income
        BigDecimal netIncome = totalRevenue.subtract(totalExpenses);
        System.out.println("SUMMARY");
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "Total Revenue", totalRevenue.setScale(2, RoundingMode.HALF_UP));
        System.out.printf("%-30s %10s%n", "Total Expenses", totalExpenses.setScale(2, RoundingMode.HALF_UP));
        System.out.println("-".repeat(40));
        System.out.printf("%-30s %10s%n", "NET INCOME", netIncome.setScale(2, RoundingMode.HALF_UP));
    }
}

