# Run 9

Creating a simple virtual machine (VM) runtime with an operand stack and a context stack in Java is an interesting challenge. I'll guide you through a basic structure of how such a VM might be implemented. We'll opt for simplicity, and the VM will support only basic operations like integer arithmetic.

### Step 1: Define the core data structures

We'll start by defining the `Frame` class, which holds information about function calls, including a local variable array and the instruction pointer.

```java
class Frame {
    final int[] localVariables;
    int instructionPointer;

    Frame(int maxLocals) {
        this.localVariables = new int[maxLocals];
        this.instructionPointer = 0;
    }
}
```

### Step 2: Define the Virtual Machine class

The VM class will contain an operand stack and a context stack of frames. We'll also define a simple instruction set for our VM's bytecode.

```java
import java.util.Stack;

public class SimpleVM {
    private final Stack<Integer> operandStack;
    private final Stack<Frame> contextStack;

    public SimpleVM() {
        operandStack = new Stack<>();
        contextStack = new Stack<>();
    }

    public void push(int value) {
        operandStack.push(value);
    }

    public int pop() {
        return operandStack.pop();
    }

    public void pushFrame(Frame frame) {
        contextStack.push(frame);
    }

    public Frame popFrame() {
        return contextStack.pop();
    }

    public Frame currentFrame() {
        return contextStack.peek();
    }

    public void execute(byte[] bytecode) {
        while (!contextStack.isEmpty()) {
            Frame frame = currentFrame();
            if (frame.instructionPointer >= bytecode.length) {
                popFrame();
                continue;
            }

            byte instruction = bytecode[frame.instructionPointer++];

            switch (instruction) {
                case Opcode.PUSH_CONST: {
                    int value = (bytecode[frame.instructionPointer++] << 24) |
                                ((bytecode[frame.instructionPointer++] & 0xFF) << 16) |
                                ((bytecode[frame.instructionPointer++] & 0xFF) << 8) |
                                (bytecode[frame.instructionPointer++] & 0xFF);
                    push(value);
                    break;
                }
                case Opcode.ADD: {
                    int b = pop();
                    int a = pop();
                    push(a + b);
                    break;
                }
                case Opcode.SUB: {
                    int b = pop();
                    int a = pop();
                    push(a - b);
                    break;
                }
                case Opcode.MUL: {
                    int b = pop();
                    int a = pop();
                    push(a * b);
                    break;
                }
                case Opcode.DIV: {
                    int b = pop();
                    int a = pop();
                    if (b == 0) {
                        throw new ArithmeticException("Division by zero");
                    }
                    push(a / b);
                    break;
                }
                case Opcode.LOAD_VAR: {
                    int index = bytecode[frame.instructionPointer++];
                    push(frame.localVariables[index]);
                    break;
                }
                case Opcode.STORE_VAR: {
                    int index = bytecode[frame.instructionPointer++];
                    frame.localVariables[index] = pop();
                    break;
                }
                case Opcode.RET: {
                    popFrame();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid bytecode encountered: " + instruction);
                }
            }
        }
    }

    public static void main(String[] args) {
        // Simple test of the VM
        SimpleVM vm = new SimpleVM();

        // Example bytecode: Push 2, Push 3, Multiply, Push 5, Add
        byte[] bytecode = new byte[] {
                Opcode.PUSH_CONST, 0, 0, 0, 2,
                Opcode.PUSH_CONST, 0, 0, 0, 3,
                Opcode.MUL,
                Opcode.PUSH_CONST, 0, 0, 0, 5,
                Opcode.ADD,
                Opcode.RET
        };

        Frame initialFrame = new Frame(10);
        vm.pushFrame(initialFrame);
        vm.execute(bytecode);

        int result = vm.pop();
        System.out.println("Result: " + result);  // Expect Result: 11 ((2 * 3) + 5)
    }
}
```

### Step 3: Define the instruction set

Next, let's define the bytecode instruction set we'll use in our VM.

```java
class Opcode {
    public static final byte PUSH_CONST = 0x01;
    public static final byte ADD = 0x02;
    public static final byte SUB = 0x03;
    public static final byte MUL = 0x04;
    public static final byte DIV = 0x05;
    public static final byte LOAD_VAR = 0x06;
    public static final byte STORE_VAR = 0x07;
    public static final byte RET = 0x08;
}
```

### Step 4: Test the Virtual Machine

The `main` method in `SimpleVM` provides a small program to test the VM. This VM will:

1. **Push 2** onto the operand stack.
2. **Push 3** onto the operand stack.
3. **Multiply** the top two elements on the stack.
4. **Push 5** onto the stack.
5. **Add** the result of the multiplication and 5.

You will see the output: `Result: 11`.

### Extending the VM

The VM as presented is very rudimentary. You can extend this by adding support for loops, branching, function calls, and more complex data types. Additionally, handling exceptions and adding a garbage collector could make this VM more sophisticated.

### Conclusion

This basic example should give you a good idea of how to start building a VM from scratch. By separating the operand stack and the execution context (frames), you'll have the flexibility to manage operations and function calls in a stack-based machine.