Creating a virtual machine (VM) runtime in Java with an operand stack and a context stack for frames involves some interesting tasks. Below is a simplified implementation outline that will get you started. This implementation includes an `OperandStack`, a `Frame` that holds local variables and an operand stack, and a `VM` class that orchestrates the execution.

### Step 1: Define the Operand Stack

We'll start by defining the `OperandStack` class.

```java
import java.util.Stack;

public class OperandStack {
    private Stack<Integer> stack;

    public OperandStack() {
        this.stack = new Stack<>();
    }

    public void push(int value) {
        stack.push(value);
    }

    public int pop() {
        return stack.pop();
    }

    public int peek() {
        return stack.peek();
    }

    public boolean isEmpty() {
        return stack.isEmpty();
    }

    public int size() {
        return stack.size();
    }
}
```

### Step 2: Define the Frame

A frame holds local variables and an operand stack specifically for that method invocation.

```java
public class Frame {
    private int[] localVariables;
    private OperandStack operandStack;

    public Frame(int localVariableCount, int operandStackSize) {
        this.localVariables = new int[localVariableCount];
        this.operandStack = new OperandStack();
    }

    // Methods to manipulate local variables.
    public void setLocalVariable(int index, int value) {
        localVariables[index] = value;
    }

    public int getLocalVariable(int index) {
        return localVariables[index];
    }

    // Methods to manipulate operand stack.
    public OperandStack getOperandStack() {
        return operandStack;
    }
}
```

### Step 3: Define the Virtual Machine (VM)

Next, create the `VM` class that manages the frames and their execution.

```java
import java.util.Stack;

public class VM {
    private Stack<Frame> stack;

    public VM() {
        this.stack = new Stack<>();
    }

    public void pushFrame(Frame frame) {
        stack.push(frame);
    }

    public Frame popFrame() {
        return stack.pop();
    }

    public Frame currentFrame() {
        return stack.peek();
    }

    public boolean isEmpty() {
        return stack.isEmpty();
    }

    public void execute() {
        // Typically, you would implement an instruction set here
        // and write logic to decode and execute instructions.
        // For now, we'll provide a very simple example.
        while (!isEmpty()) {
            Frame frame = currentFrame();
            OperandStack operandStack = frame.getOperandStack();

            // Example case: two integer addition operation
            if (operandStack.size() >= 2) {
                int operand1 = operandStack.pop();
                int operand2 = operandStack.pop();
                int result = operand1 + operand2;

                operandStack.push(result);
            }

            // You've just executed the current frame's instruction set;
            // Typically you'd remove the frame if it has completed all instructions.
            popFrame();
        }
    }
}
```

### Step 4: Create a Sample Use Case

Here is an example of how to use the `VM` to add two numbers:

```java
public class Main {
    public static void main(String[] args) {
        // Create a VM instance
        VM vm = new VM();

        // Create a frame with local variables and operand stack
        Frame frame = new Frame(10, 10);

        // Push two numbers onto the operand stack
        frame.getOperandStack().push(10);
        frame.getOperandStack().push(20);

        // Push the frame onto the VM stack
        vm.pushFrame(frame);

        // Execute (in this case it will perform the addition and leave the result on the stack)
        vm.execute();

        // Get the result from the operand stack
        if (!frame.getOperandStack().isEmpty()) {
            System.out.println("Result: " + frame.getOperandStack().pop());  // Should output "Result: 30"
        } else {
            System.out.println("No result on stack");
        }
    }
}
```

### Conclusion:
This code provides a basic framework for a simple virtual machine runtime that uses an operand stack and a context stack for executing frames. This example uses integer addition as demonstration, but in a more complex implementation, you'd have a detailed set of bytecode instructions and a much richer execution environment.

This is, of course, a simplistic implementation and can be considerably extended depending on requirements (e.g., additional operations, handling of different data types, support for function calls, etc.).