Creating a simple virtual machine (VM) runtime in Java is an involved task, but I can guide you through implementing the basics, including an operand stack and a context stack for frames. Below is a simplified example that demonstrates how you can create these components.

### 1. Define the Operand Stack

The operand stack is where we will manipulate the values during the computation (i.e., push, pop, etc.). 

```java
import java.util.Stack;

public class OperandStack {

    private Stack<Integer> stack;

    public OperandStack() {
        this.stack = new Stack<>();
    }

    public void push(int value) {
        stack.push(value);
    }

    public int pop() {
        return stack.pop();
    }

    public int peek() {
        return stack.peek();
    }

    public boolean isEmpty() {
        return stack.isEmpty();
    }

    public int size() {
        return stack.size();
    }
}
```

### 2. Define the Frame Context

Each method call or execution block will have its own frame context, which holds its own operand stack and local variables.

```java
import java.util.HashMap;
import java.util.Map;

public class Frame {

    private OperandStack operandStack;
    private Map<Integer, Integer> localVariables;

    public Frame() {
        this.operandStack = new OperandStack();
        this.localVariables = new HashMap<>();
    }

    public OperandStack getOperandStack() {
        return operandStack;
    }

    public void setLocalVariable(int index, int value) {
        localVariables.put(index, value);
    }

    public int getLocalVariable(int index) {
        return localVariables.getOrDefault(index, 0);
    }
}
```

### 3. Define the Context Stack

The context stack manages the stack of frames — typically a method call stack.

```java
import java.util.Stack;

public class ContextStack {

    private Stack<Frame> frameStack;

    public ContextStack() {
        this.frameStack = new Stack<>();
    }

    public void pushFrame(Frame frame) {
        frameStack.push(frame);
    }

    public Frame popFrame() {
        return frameStack.pop();
    }

    public Frame peekFrame() {
        return frameStack.peek();
    }

    public boolean isEmpty() {
        return frameStack.isEmpty();
    }
}
```

### 4. Implement the Virtual Machine

Finally, let's combine these components into a simple virtual machine.

```java
public class VirtualMachine {

    private ContextStack contextStack;

    public VirtualMachine() {
        this.contextStack = new ContextStack();
    }

    public void execute() {
        while (!contextStack.isEmpty()) {
            Frame currentFrame = contextStack.peekFrame();
            OperandStack operandStack = currentFrame.getOperandStack();

            // Example: Perform a basic operation like addition
            if (operandStack.size() >= 2) {
                int b = operandStack.pop();
                int a = operandStack.pop();
                int result = a + b;

                operandStack.push(result);
            }

            // Example: Pop the top frame when done
            contextStack.popFrame();
        }
    }

    public void addFrame(Frame frame) {
        contextStack.pushFrame(frame);
    }

    public static void main(String[] args) {
        VirtualMachine vm = new VirtualMachine();

        // Create a basic frame
        Frame frame = new Frame();

        // Push some values onto the operand stack
        frame.getOperandStack().push(10);
        frame.getOperandStack().push(20);

        // Add this frame to the context stack
        vm.addFrame(frame);

        // Execute the VM
        vm.execute();

        // Check the result of the computation
        System.out.println("Result: " + frame.getOperandStack().peek());  // Should print 30.
    }
}
```

### Explanation:

- **Operand Stack:** This acts as the primary workspace for the virtual machine's execution, where individual operations work on the top of the stack (e.g., for addition, two values are popped, added together, and then the result is pushed back).
  
- **Frame:** This represents the execution context for a piece of code, including its own operand stack and local variables.
  
- **Context Stack:** Manages the frames, typically corresponding to the stack of method calls.

- **Virtual Machine:** This integrates everything, allowing you to push frames onto the stack, execute operations, and retrieve results.

### Extending the VM:

- **Instruction Set:** You could define a set of bytecode instructions (add, subtract, multiply, etc.) and an interpreter loop within the `execute()` method.

- **Advanced Operand Stack:** You might extend the OperandStack to deal with multiple data types, not just integers.

- **Advanced Frame Management:** For functions or methods, you could handle return addresses and return values.

This is a simplistic outline, and a full-fledged VM would include many additional features like exception handling, security checks, and garbage collection, but this should provide a basic foundation.