FastAPI, a modern Python web framework, is often used with database modeling to create robust APIs. Here's how it's typically done:

1. Database Choice:
FastAPI doesn't enforce a specific database. You can choose SQL (like PostgreSQL, MySQL, SQLite) or NoSQL databases.
For SQL databases, SQLAlchemy is a common choice as an ORM (Object-Relational Mapper).
SQLModel, created by the same author as FastAPI, is designed to work seamlessly with it.
2. Data Models:
Pydantic: FastAPI uses Pydantic for data validation and serialization. You define data models using Pydantic, which are then used to validate incoming requests and structure responses.
SQLModel: SQLModel extends Pydantic to define database tables directly as Python classes. This allows you to define both data structures and database schemas in one place.
SQLAlchemy: If using SQLAlchemy, you define models that map to database tables.
3. Database Interaction:
SQLAlchemy: With SQLAlchemy, you create an engine that connects to your database. Then you define sessions to interact with the database.
SQLModel: SQLModel simplifies this process by providing its own engine and session management.
You typically create a dependency that yields a database session for each request to ensure proper database interaction.
4. ORM vs. Raw SQL:
ORMs like SQLAlchemy and SQLModel allow you to interact with databases using Python objects instead of writing raw SQL queries, making code more readable and maintainable.
You can still use raw SQL if needed.
5. FastAPI Integration:
FastAPI endpoints use Pydantic models to define request and response structures.
You can inject database sessions as dependencies into your endpoints.
FastAPI automatically generates API documentation based on your Pydantic models.
6. Example with SQLModel:
```Python

   from typing import Optional
   from fastapi import FastAPI, Depends
   from sqlmodel import Field, Session, SQLModel, create_engine, select

   class Hero(SQLModel, table=True):
       id: Optional[int] = Field(default=None, primary_key=True)
       name: str
       secret_name: str
       age: Optional[int] = Field(default=None, index=True)

   sqlite_file_name = "database.db"
   sqlite_url = f"sqlite:///{sqlite_file_name}"
   connect_args = {"check_same_thread": False}
   engine = create_engine(sqlite_url, connect_args=connect_args)

   def create_db_and_tables():
       SQLModel.metadata.create_all(engine)

   def get_session():
       with Session(engine) as session:
           yield session

   SessionDep = Depends(get_session)
   app = FastAPI()

   @app.on_event("startup")
   def on_startup():
       create_db_and_tables()

   @app.get("/heroes/")
   def read_heroes(session: SessionDep):
       heroes = session.exec(select(Hero)).all()
       return heroes
```

7. Key Considerations:
Dependencies: Use FastAPI's dependency injection to manage database sessions effectively.
Error Handling: FastAPI automatically handles validation errors based on Pydantic models.
Documentation: FastAPI generates interactive API documentation.
Testing: Write tests for your database interactions.
FastAPI's integration with Pydantic and ORMs simplifies database modeling, making it easier to build efficient and maintainable APIs.