describe how you would use python's dict data structure to build a symbol table for a small language compiler

## Phi3


Building a symbol table using Python's dictionary data structure for a small language compiler involves several 
steps. The symbol table will be used to keep track of variables, their types, and scopes during the compilation 
process. Here's how you can implement it:

1. Define basic classes
Firstly, create two essential classes - `Symbol` and `Scope`.
```python
class Symbol:
    def __init__(self, name, symbol_type):
        self.name = name
        self.symbol_type = symbol_type

class Scope:
    def __init__(self, parent=None):
        self.parent = parent
        self.scopes = []  # list of symbols in this scope
```
2. Initialize the global symbol table
Create a class `SymbolTable` which will manage all scopes and keep track of symbols globally using Python's 
dictionary data structure.
```python
class SymbolTable:
    def __init__(self):
        self.symbols = {}  # Key: symbol_name, Value: (Scope object)
```
3. Add new symbols to the table
Define a function `addSymbol` in `SymbolTable`, which adds symbols according to their scopes and types. This 
method ensures that overlapping names are handled gracefully by using the scope's parent as reference if 
necessary.
```python
class SymbolTable:
    # ... existing code ...
    
    def addSymbol(self, symbol_name, symbol_type):
        current_scope = self.symbols.get(symbol_name)
        
        if not current_scope:  # If the symbol doesn't exist in any scope yet, create it at global level
            new_scope = Scope()
            self.symbols[symbol_name] = (new_scope, SymbolType(symbol_type))
            return new_scope
        
        # Else check if we are dealing with an inner symbol
        else:
            scope = current_scope[0]  # parent Scope object
            
            while scope.parent and scope.name in self.symbols:  # Check all parents until a free name is found
                scope = scope.parent
                
            if not scope.parent or scope.name not in self.symbols:  # Add the symbol to current scope if there's 
no conflict with any parent scope
                new_scope = Scope(parent=scope)
                self.symbols[symbol_name] = (new_scope, SymbolType(symbol_type))
            else:
                raise NameError('Symbol name already exists in this or one of the parent scopes')
```
4. Retrieve symbols from the table
Define a method `getSymbol` to retrieve symbols by their names according to their respective scope and type 
information.
```python
class SymbolTable:
    # ... existing code ...
    
    def getSymbol(self, symbol_name):
        if symbol_name in self.symbols:
            return self.symbols[symbol_name]
        else:
            raise NameError('No such symbol')
```

With this implementation, you've created a basic symbol table using Python dictionaries that can track variables 
and their types for a small language compiler. You can extend it further by implementing methods to handle more 
complex scenarios like type inference or variable renaming during optimization stages of the compilation process.

