# Introduction to C for Java and Python Programmers

As a Java or Python programmer, you'll find C both familiar and different. C is a procedural language that directly influenced both Java and Python, but it's lower-level, giving you more direct control over memory and hardware. Here's an introduction to help you bridge the gap.

## Core Characteristics of C

C is a compiled, statically-typed language with manual memory management. Unlike Java's JVM or Python's interpreter, C code compiles directly to machine code for your specific hardware. There's no garbage collection - you allocate and free memory yourself.

## Basic Syntax and Data Types

C syntax will look familiar with some key differences:
```c
#include <stdio.h>  // Header files use angle brackets

int main() {  // Every C program needs a main function
    // Basic data types
    int number = 42;          // Integer
    float decimal = 3.14;     // Single-precision floating point
    double precise = 3.14159; // Double-precision floating point
    char letter = 'A';        // Single character
    
    // C has no built-in boolean type (before C99)
    // Traditionally 0 is false, anything else is true
    int is_valid = 1;  // true
    
    // Output
    printf("Number: %d, Decimal: %f\n", number, decimal);
    
    return 0;  // Return value from main indicates program status
}
```

## Control Structures

Control structures are similar to Java:
```c
int value = 10;

// If-else statement
if (value > 5) {
    printf("Value is greater than 5\n");
} else if (value == 5) {
    printf("Value is equal to 5\n");
} else {
    printf("Value is less than 5\n");
}

// While loop
int counter = 0;
while (counter < 5) {
    printf("Counter: %d\n", counter);
    counter++;
}

// For loop
for (int i = 0; i < 5; i++) {
    printf("Index: %d\n", i);
}

// Switch statement
switch (value) {
    case 5:
        printf("Value is 5\n");
        break;
    case 10:
        printf("Value is 10\n");
        break;
    default:
        printf("Value is something else\n");
}
```

## Functions

Functions in C are straightforward but require explicit type declarations:
```c
// Function declaration (prototype)
int add(int a, int b);

int main() {
    int result = add(5, 3);
    printf("5 + 3 = %d\n", result);
    return 0;
}

// Function definition
int add(int a, int b) {
    return a + b;
}
```

## Arrays

Arrays in C are fixed-size and zero-indexed:
```c
// Array declaration and initialization
int numbers[5] = {1, 2, 3, 4, 5};

// Accessing elements
printf("Third element: %d\n", numbers[2]);  // Prints 3

// Array without size (compiler calculates)
char name[] = "Hello";  // Creates a 6-element array (including null terminator)

// Multi-dimensional arrays
int matrix[3][3] = {
    {1, 2, 3},
    {4, 5, 6},
    {7, 8, 9}
};

printf("matrix[1][2] = %d\n", matrix[1][2]);  // Prints 6
```

## Strings

Unlike Java and Python, C has no built-in string type. Strings are arrays of characters terminated by a null character (`\0`):
```c
// String declaration
char greeting[] = "Hello, World!";  // Compiler adds the null terminator

// String functions require the string.h header
#include <string.h>

int main() {
    char str1[50] = "Hello";
    char str2[] = ", World!";
    
    // String length
    printf("Length: %lu\n", strlen(str1));  // 5
    
    // String concatenation
    strcat(str1, str2);
    printf("Concatenated: %s\n", str1);  // "Hello, World!"
    
    // String comparison
    if (strcmp(str1, "Hello, World!") == 0) {
        printf("Strings are equal\n");
    }
    
    return 0;
}
```

## Structs

Structs let you create custom data types by grouping related variables:
```c
// Struct definition
struct Person {
    char name[50];
    int age;
    float height;
};

int main() {
    // Creating a struct variable
    struct Person person1;
    
    // Assigning values
    strcpy(person1.name, "Alice");
    person1.age = 30;
    person1.height = 1.75;
    
    // Alternative initialization syntax
    struct Person person2 = {"Bob", 25, 1.82};
    
    // Accessing struct members
    printf("Name: %s, Age: %d, Height: %.2f\n", 
           person1.name, person1.age, person1.height);
    
    return 0;
}
```

## Pointers

Pointers are perhaps the most distinctive feature of C. They store memory addresses:
```c
int main() {
    int x = 10;
    int *ptr = &x;  // ptr stores the address of x
    
    printf("Value of x: %d\n", x);       // 10
    printf("Address of x: %p\n", &x);    // Memory address
    printf("Value of ptr: %p\n", ptr);   // Same memory address
    printf("Value at *ptr: %d\n", *ptr); // 10 (dereferencing)
    
    // Modifying value through pointer
    *ptr = 20;
    printf("New value of x: %d\n", x);   // 20
    
    return 0;
}
```

Pointers are essential for:
1. Dynamic memory allocation
2. Passing large data structures efficiently
3. Creating complex data structures like linked lists

## Dynamic Memory Allocation

Unlike Java and Python, C requires manual memory management:
```c
#include <stdlib.h>  // Required for malloc and free

int main() {
    // Allocate memory for an integer
    int *ptr = (int *)malloc(sizeof(int));
    
    // Check if allocation was successful
    if (ptr == NULL) {
        printf("Memory allocation failed\n");
        return 1;
    }
    
    // Use the allocated memory
    *ptr = 42;
    printf("Value: %d\n", *ptr);
    
    // Free the memory when done
    free(ptr);
    
    // Allocate an array of integers
    int *array = (int *)malloc(5 * sizeof(int));
    
    if (array != NULL) {
        for (int i = 0; i < 5; i++) {
            array[i] = i * 10;
            printf("array[%d] = %d\n", i, array[i]);
        }
        free(array);
    }
    
    return 0;
}
```

## Key Differences to Remember

1. **Manual Memory Management**: Unlike Java and Python, C has no garbage collection. You must free any memory you allocate.
2. **No Objects or Classes**: C is procedural, not object-oriented.
3. **No Exception Handling**: C uses return values to indicate errors.
4. **Pointers**: Direct memory manipulation is both C's power and danger.
5. **No Built-in Data Structures**: No lists, dictionaries, or sets; you build these yourself.

C is powerful because it's close to the hardware, but this requires more attention to detail. Start with small programs, and always check your code for memory leaks and pointer errors.

