# JSON Fundamentals

JavaScript Object Notation (JSON) is a standard text-based format for representing structured data 
based on JavaScript object syntax. 
It is commonly used for transmitting data in web applications (e.g., sending some data 
from the server to the client, so it can be displayed on a web page, or vice versa). 
You'll come across it quite often, so in this article, we give you all you need to 
work with JSON using JavaScript, including parsing JSON so you can access data within it, and creating JSON.

## Data Types

JSON (JavaScript Object Notation) is a __text-based data exchange format__. 
It is a collection of key-value pairs where the key must be a string type, and the value can be of any of the following types:

- Number
- String
- Boolean
- Array
- Object
- null

A couple of important rules to note:

- In the JSON data format, the keys must be enclosed in double quotes.
- The key and value must be separated by a colon (:) symbol.
- There can be multiple key-value pairs. Two key-value pairs must be separated by a comma (,) symbol.
- No comments (// or /* */) are allowed in JSON data. (But you can get around that, if you're curious.)

## Usage

Often, you may read or write it to files, for configuration purposes, small data sets, and useful structured data.
It is a common, go-to data format. Most languages have great support for reading and writing it, like Java and Python.

JSON exists as a string — useful when you want to transmit data across a network. 
It needs to be converted to a native object when you want to access the data within a program. 

Example of [python reading of file containing JSON.

```json
# Python program to read
# json file

import json

# Opening JSON file
f = open('data.json')

# returns JSON object as 
# a dictionary
data = json.load(f)

# Iterating through the json
# list
for i in data['emp_details']:
    print(i)

# Closing file
f.close()
```

Writing JSON:

`data` is a Python dictionary. It needs to be encoded as JSON before writing.

Use this for maximum compatibility (Python3 & 2, if you must):

```python
import json
with open('data.json', 'w') as f:
    json.dump(data, f)
```

On a modern system (i.e. Python 3 and UTF-8 support), you can write a nicer file using:

```python
import json
with open('data.json', 'w', encoding='utf-8') as f:
    json.dump(data, f, ensure_ascii=False, indent=4)
```

Check out: [JSON (JavaScript Object Notation)](https://www.json.org/json-en.html)
It has a great set of syntax flow diagrams you can use to under stand it.

## JSON Examples 

A simple object

```json
{
    "fruit": "Apple",
    "size": "Large",
    "color": "Red"
}
```

Maybe something that came from a bloggin system.

```json
{
  "title": "New Blog Post",
  "content": "This is the content of the blog post...",
  "publishedDate": "2023-08-25T15:00:00Z",
  "author": {
    "username": "authoruser",
    "email": "author@example.com"
  },
  "tags": ["Technology", "Programming"]
}
```

Or this one, a example of something whihc might have come from a database:

```json
{
  "patientName": "Jane Doe",
  "dateOfBirth": "1985-02-15",
  "bloodType": "A+",
  "allergies": ["Pollen", "Penicillin"],
  "conditions": ["Hypertension", "Diabetes"],
  "medications": ["Lisinopril", "Metformin"],
  "emergencyContact": {
    "username": "emergencyuser",
    "email": "johnDoe@gmale.com"
  }
}
```

An array:

```json
{
"employees":[
  {"firstName":"John", "lastName":"Doe"}, 
  {"firstName":"Anna", "lastName":"Smith"},
  {"firstName":"Peter", "lastName":"Jones"}
]
}
```
















