# Explaining APIs to Beginner Programmers

Here is an explaination APIs to beginner programmers with examples in both Java and Python. 
APIs (Application Programming Interfaces) are indeed a fundamental concept for new programmers to understand.

## What is an API?

An API is like a contract between different software components that defines how they should interact. 
Think of it as a menu at a restaurant - you don't need to know how the kitchen prepares the food, 
you just need to know what you can order and how to place that order.

## A Great Public API for Beginners

The **OpenWeatherMap API** is perfect for beginners because:
- It's free to use (with registration)
- It has straightforward endpoints
- The responses are easy to understand
- It's well-documented
- It works well with both Java and Python

Let me show you how to use this API in both languages.

### Step 1: Get an API Key

First, register at [OpenWeatherMap](https://openweathermap.org/api) to get a free API key.

### Step 2: Make API Calls

#### Python Example:

```python
import requests

def get_weather(city, api_key):
    """
    Get current weather for a city using OpenWeatherMap API
    """
    base_url = "https://api.openweathermap.org/data/2.5/weather"
    
    # Parameters for our API request
    params = {
        "q": city,
        "appid": api_key,
        "units": "metric"  # For Celsius
    }
    
    # Make the API call
    response = requests.get(base_url, params=params)
    
    # Check if the request was successful
    if response.status_code == 200:
        data = response.json()  # Convert response to Python dictionary
        
        # Extract relevant information
        weather_description = data["weather"][0]["description"]
        temperature = data["main"]["temp"]
        humidity = data["main"]["humidity"]
        
        print(f"Weather in {city}:")
        print(f"Description: {weather_description}")
        print(f"Temperature: {temperature}°C")
        print(f"Humidity: {humidity}%")
    else:
        print(f"Error: {response.status_code}")
        print(response.text)

# Example usage
if __name__ == "__main__":
    city = "London"
    api_key = "your_api_key_here"  # Replace with actual API key
    get_weather(city, api_key)
```

#### Java Example:

```java
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import org.json.JSONObject;  // Requires JSON library (e.g., org.json)

public class WeatherAPI {
    
    public static void main(String[] args) {
        String city = "London";
        String apiKey = "your_api_key_here";  // Replace with actual API key
        
        try {
            getWeather(city, apiKey);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    public static void getWeather(String city, String apiKey) throws Exception {
        // Create URL with parameters
        String encodedCity = URLEncoder.encode(city, StandardCharsets.UTF_8);
        String urlString = "https://api.openweathermap.org/data/2.5/weather" +
                "?q=" + encodedCity +
                "&appid=" + apiKey +
                "&units=metric";
        
        URL url = new URL(urlString);
        
        // Open connection
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod("GET");
        
        // Check response code
        int responseCode = connection.getResponseCode();
        
        if (responseCode == 200) {
            // Read response
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(connection.getInputStream()));
            StringBuilder response = new StringBuilder();
            String line;
            
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            reader.close();
            
            // Parse JSON
            JSONObject data = new JSONObject(response.toString());
            JSONObject main = data.getJSONObject("main");
            JSONObject weather = data.getJSONArray("weather").getJSONObject(0);
            
            // Extract information
            String description = weather.getString("description");
            double temperature = main.getDouble("temp");
            int humidity = main.getInt("humidity");
            
            System.out.println("Weather in " + city + ":");
            System.out.println("Description: " + description);
            System.out.println("Temperature: " + temperature + "°C");
            System.out.println("Humidity: " + humidity + "%");
        } else {
            System.out.println("Error: " + responseCode);
        }
        
        connection.disconnect();
    }
}
```

## Key API Concepts to Teach

1. **API Endpoints**: The specific URLs that accept requests
2. **Authentication**: How to use API keys for access
3. **Request Parameters**: How to customize what data you want
4. **HTTP Methods**: GET, POST, PUT, DELETE and their purposes
5. **Response Formats**: Usually JSON or XML
6. **Status Codes**: Understanding what 200, 404, 500, etc. mean
7. **Error Handling**: What to do when things go wrong

## Additional API Suggestions for Beginners

If weather doesn't interest you, here are other beginner-friendly public APIs:

1. **NASA APOD API**: Get astronomy pictures of the day
2. **PokeAPI**: Information about Pokémon
3. **JSONPlaceholder**: Fake data for testing and prototyping
4. **Open Trivia Database**: Random trivia questions
5. **Dog CEO**: Random dog images

# API Concepts: A Comprehensive Study Guide for Beginners

## Introduction

Application Programming Interfaces (APIs) have become an essential skill for modern programmers. They allow different software systems to communicate with each other, enabling developers to leverage existing services rather than building everything from scratch. This study guide introduces key API concepts with practical examples in both Java and Python to help beginners develop a solid understanding of how APIs work and how to use them effectively.

## 1. API Endpoints

### What are API Endpoints?

API endpoints are specific URLs that receive API requests. Think of them as the "front door" to a specific service or resource provided by an API. Each endpoint typically represents a particular function or resource in the system and usually follows a hierarchical structure.

### Understanding Endpoint Structure

API endpoints typically consist of:
- A base URL (e.g., `https://api.openweathermap.org`)
- API version (e.g., `/data/2.5/`)
- The specific resource or function (e.g., `/weather` or `/forecast`)

For example, OpenWeatherMap offers different endpoints for different weather data:
```
https://api.openweathermap.org/data/2.5/weather    // Current weather data
https://api.openweathermap.org/data/2.5/forecast   // 5-day forecast
https://api.openweathermap.org/data/2.5/onecall    // Current, minute forecast, hourly forecast, and daily forecast
```

### How to Navigate API Documentation

When learning a new API, always start with its documentation. Good API documentation will list all available endpoints along with:
- What the endpoint does
- Required parameters
- Optional parameters
- Response format
- Example requests and responses

For instance, the OpenWeatherMap API documentation will tell you that to get current weather data, you need to use the `/weather` endpoint with a city name or coordinates.

### RESTful API Design

Many modern APIs follow REST (Representational State Transfer) principles, which organize endpoints around resources. RESTful APIs typically use:
- Nouns (not verbs) in endpoint paths to represent resources
- HTTP methods to indicate actions on those resources
- Consistent URL patterns

For example, a RESTful API for a blog might have endpoints like:
```
GET /posts                  // Get all posts
GET /posts/123              // Get a specific post
POST /posts                 // Create a new post
PUT /posts/123              // Update post 123
DELETE /posts/123           // Delete post 123
GET /posts/123/comments     // Get comments for post 123
```

### Practice Exercise

1. Visit the documentation for a public API (like OpenWeatherMap, GitHub, or Spotify).
2. Identify at least three different endpoints.
3. For each endpoint, note what resource it represents and what information it provides.
4. Try to identify the pattern in how the endpoints are structured.

## 2. Authentication

### Why Authentication Matters

Authentication is the process of verifying who is making the request to an API. It's crucial for:
- Protecting private or sensitive data
- Preventing abuse or misuse of the API
- Tracking usage for rate limiting or billing purposes
- Associating requests with specific users or applications

Without authentication, anyone could potentially access sensitive data or abuse an API service, causing performance issues or excessive costs.

### Common Authentication Methods

#### API Keys

The simplest form of authentication, typically sent as a query parameter or header:

**Python Example:**
```python
import requests

api_key = "your_api_key_here"
url = f"https://api.openweathermap.org/data/2.5/weather?q=London&appid={api_key}"

response = requests.get(url)
```

**Java Example:**
```java
import java.net.URL;
import java.net.HttpURLConnection;

String apiKey = "your_api_key_here";
URL url = new URL("https://api.openweathermap.org/data/2.5/weather?q=London&appid=" + apiKey);

HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");
```

#### OAuth 2.0

A more complex authentication protocol used when an application needs to access user data on another service (like logging in with Google):

**Python Example:**
```python
import requests
from requests_oauthlib import OAuth2Session

client_id = "your_client_id"
client_secret = "your_client_secret"
redirect_uri = "your_redirect_uri"
scope = ["profile", "email"]  # The permissions you're requesting

# Step 1: Redirect user to authorization URL
oauth = OAuth2Session(client_id, redirect_uri=redirect_uri, scope=scope)
authorization_url, state = oauth.authorization_url("https://accounts.google.com/o/oauth2/auth")

# Step 2: Get authorization code from redirect and exchange for token
token = oauth.fetch_token(
    "https://accounts.google.com/o/oauth2/token",
    client_secret=client_secret,
    authorization_response=redirect_response
)

# Step 3: Use token to access API
response = oauth.get("https://www.googleapis.com/oauth2/v1/userinfo")
```

**Java Example (using Spring OAuth):**
```java
@RestController
public class OAuthController {
    @GetMapping("/login")
    public String login() {
        return "redirect:https://accounts.google.com/o/oauth2/auth?client_id=YOUR_CLIENT_ID&redirect_uri=YOUR_REDIRECT_URI&scope=profile email&response_type=code";
    }
    
    @GetMapping("/callback")
    public String callback(@RequestParam String code) {
        // Exchange code for token
        // Use token to access API
    }
}
```

#### Bearer Tokens (JWT)

JSON Web Tokens are a compact, URL-safe means of representing claims between two parties:

**Python Example:**
```python
import requests

token = "your_jwt_token"
headers = {
    "Authorization": f"Bearer {token}"
}

response = requests.get("https://api.example.com/resource", headers=headers)
```

**Java Example:**
```java
URL url = new URL("https://api.example.com/resource");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");
connection.setRequestProperty("Authorization", "Bearer " + token);
```

### Securing API Credentials

Always protect your API credentials:

1. **Never hardcode credentials** in your source code, especially in repositories that might become public.
2. **Use environment variables** to store sensitive information:

   **Python:**
   ```python
   import os
   api_key = os.environ.get("API_KEY")
   ```

   **Java:**
   ```java
   String apiKey = System.getenv("API_KEY");
   ```

3. **Use configuration files** that are excluded from version control:

   **Python (config.py):**
   ```python
   # Add config.py to .gitignore
   API_KEY = "your_key_here"
   ```

   **Java (config.properties):**
   ```
   # Add config.properties to .gitignore
   api.key=your_key_here
   ```

4. **Implement credential rotation** for production applications, changing keys periodically.

5. **Use least privilege** - only request the permissions your application needs.

### Practice Exercise

1. Register for a free API key from OpenWeatherMap or another public API.
2. Create a small application that uses your API key to make a request.
3. Implement at least two different methods of storing your API key securely.
4. Try implementing a simple request to an API that uses OAuth (like GitHub or Spotify).

## 3. Request Parameters

### Types of Parameters

Parameters allow you to customize API requests by providing additional data. Different parameter types serve different purposes:

#### Query Parameters

Added to the URL after a question mark (`?`) and separated by ampersands (`&`):

**Python:**
```python
import requests

params = {
    "q": "London",
    "units": "metric",
    "appid": "your_api_key"
}

response = requests.get("https://api.openweathermap.org/data/2.5/weather", params=params)

# Resulting URL: https://api.openweathermap.org/data/2.5/weather?q=London&units=metric&appid=your_api_key
```

**Java:**
```java
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

String city = URLEncoder.encode("London", StandardCharsets.UTF_8);
String apiKey = "your_api_key";
String urlString = "https://api.openweathermap.org/data/2.5/weather" +
        "?q=" + city +
        "&units=metric" +
        "&appid=" + apiKey;

URL url = new URL(urlString);
```

#### Path Parameters

Embedded directly in the URL path, usually indicated in documentation with curly braces:

**Python:**
```python
import requests

user_id = "12345"
response = requests.get(f"https://api.github.com/users/{user_id}/repos")
```

**Java:**
```java
String userId = "12345";
URL url = new URL("https://api.github.com/users/" + userId + "/repos");
```

#### Header Parameters

Sent in the HTTP request headers, commonly used for authentication, content type, or custom API features:

**Python:**
```python
import requests

headers = {
    "Authorization": "Bearer your_token",
    "Content-Type": "application/json",
    "Accept-Language": "en-US"
}

response = requests.get("https://api.example.com/resource", headers=headers)
```

**Java:**
```java
URL url = new URL("https://api.example.com/resource");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");
connection.setRequestProperty("Authorization", "Bearer your_token");
connection.setRequestProperty("Content-Type", "application/json");
connection.setRequestProperty("Accept-Language", "en-US");
```

#### Request Body Parameters

Used primarily with POST, PUT, and PATCH requests to send larger amounts of data:

**Python:**
```python
import requests
import json

data = {
    "title": "New Post",
    "content": "This is the content of my new blog post.",
    "author_id": 42
}

headers = {"Content-Type": "application/json"}
response = requests.post(
    "https://api.example.com/posts",
    data=json.dumps(data),
    headers=headers
)
```

**Java:**
```java
URL url = new URL("https://api.example.com/posts");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("POST");
connection.setRequestProperty("Content-Type", "application/json");
connection.setDoOutput(true);

String jsonInputString = "{\"title\":\"New Post\",\"content\":\"This is the content of my new blog post.\",\"author_id\":42}";

try(OutputStream os = connection.getOutputStream()) {
    byte[] input = jsonInputString.getBytes(StandardCharsets.UTF_8);
    os.write(input, 0, input.length);
}
```

### Required vs. Optional Parameters

API documentation typically distinguishes between:
- **Required parameters**: Must be included for the request to succeed
- **Optional parameters**: Provide additional functionality but can be omitted

Understanding this distinction helps prevent errors and allows for more flexibility in your API calls.

### Parameter Validation

Before sending an API request, validate your parameters to avoid errors:
1. Check required parameters are present
2. Ensure parameters meet format requirements (e.g., date formats, numeric ranges)
3. Handle special characters by proper encoding

**Python Example:**
```python
def validate_weather_params(city=None, lat=None, lon=None):
    """Validate parameters for weather API call"""
    if city is None and (lat is None or lon is None):
        raise ValueError("Either city name or coordinates (lat and lon) must be provided")
    
    if lat is not None and (lat < -90 or lat > 90):
        raise ValueError("Latitude must be between -90 and 90")
    
    if lon is not None and (lon < -180 or lon > 180):
        raise ValueError("Longitude must be between -180 and 180")
```

**Java Example:**
```java
public void validateWeatherParams(String city, Double lat, Double lon) throws IllegalArgumentException {
    if (city == null && (lat == null || lon == null)) {
        throw new IllegalArgumentException("Either city name or coordinates (lat and lon) must be provided");
    }
    
    if (lat != null && (lat < -90 || lat > 90)) {
        throw new IllegalArgumentException("Latitude must be between -90 and 90");
    }
    
    if (lon != null && (lon < -180 || lon > 180)) {
        throw new IllegalArgumentException("Longitude must be between -180 and 180");
    }
}
```

### Handling Default Values

When parameters are optional, they often have default values. Understanding these defaults is important:

```python
def get_weather(city, api_key, units="metric", lang="en"):
    """
    Get weather information where:
    - units defaults to metric (Celsius)
    - language defaults to English
    """
    params = {
        "q": city,
        "appid": api_key,
        "units": units,
        "lang": lang
    }
    
    response = requests.get("https://api.openweathermap.org/data/2.5/weather", params=params)
    return response.json()
```

### Practice Exercise

1. Choose an API you're interested in and identify its different parameter types.
2. Create a function that validates parameters before making an API call.
3. Experiment with optional parameters to see how they affect the response.
4. Try sending a request with missing required parameters and observe the error.

## 4. HTTP Methods

### Understanding REST and CRUD

HTTP methods align with CRUD (Create, Read, Update, Delete) operations in a RESTful API:

| HTTP Method | CRUD Operation | Description                          |
|-------------|---------------|--------------------------------------|
| GET         | Read          | Retrieve data without modifying it    |
| POST        | Create        | Create a new resource                 |
| PUT         | Update        | Replace an entire resource            |
| PATCH       | Update        | Partially update a resource           |
| DELETE      | Delete        | Remove a resource                     |

### GET: Retrieving Data

GET requests are read-only operations used to retrieve data without changing anything on the server. They should be "safe" (don't change data) and "idempotent" (multiple identical requests have the same effect as a single request).

**Python:**
```python
import requests

# Simple GET request
response = requests.get("https://api.openweathermap.org/data/2.5/weather?q=London&appid=your_api_key")

# GET with parameters
params = {"q": "London", "appid": "your_api_key"}
response = requests.get("https://api.openweathermap.org/data/2.5/weather", params=params)

# Print response
data = response.json()
print(f"Current temperature in London: {data['main']['temp']}°C")
```

**Java:**
```java
import java.net.HttpURLConnection;
import java.net.URL;
import java.io.BufferedReader;
import java.io.InputStreamReader;

// Create URL and open connection
URL url = new URL("https://api.openweathermap.org/data/2.5/weather?q=London&appid=your_api_key");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");

// Read response
BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
StringBuilder response = new StringBuilder();
String line;
while ((line = reader.readLine()) != null) {
    response.append(line);
}
reader.close();

// Process JSON response (requires JSON library like org.json or Jackson)
// Example with org.json:
JSONObject data = new JSONObject(response.toString());
double temp = data.getJSONObject("main").getDouble("temp");
System.out.println("Current temperature in London: " + temp + "°C");
```

### POST: Creating Resources

POST requests create new resources on the server. They are not idempotent—sending the same POST request multiple times typically creates multiple resources.

**Python:**
```python
import requests
import json

# Data to create a new resource
new_post = {
    "title": "Understanding APIs",
    "body": "This is a comprehensive guide to APIs...",
    "userId": 1
}

# Set headers
headers = {"Content-Type": "application/json"}

# Make POST request
response = requests.post(
    "https://jsonplaceholder.typicode.com/posts",
    data=json.dumps(new_post),
    headers=headers
)

# Check if successful
if response.status_code == 201:  # 201 Created
    created_post = response.json()
    print(f"Created post with ID: {created_post['id']}")
else:
    print(f"Error: {response.status_code}")
    print(response.text)
```

**Java:**
```java
URL url = new URL("https://jsonplaceholder.typicode.com/posts");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("POST");
connection.setRequestProperty("Content-Type", "application/json");
connection.setDoOutput(true);

// Prepare data
String jsonData = "{\"title\":\"Understanding APIs\",\"body\":\"This is a comprehensive guide to APIs...\",\"userId\":1}";

// Send data
try (OutputStream os = connection.getOutputStream()) {
    byte[] input = jsonData.getBytes(StandardCharsets.UTF_8);
    os.write(input, 0, input.length);
}

// Check response
int responseCode = connection.getResponseCode();
if (responseCode == 201) {
    // Read and process successful response
    // ...
} else {
    // Handle error
    // ...
}
```

### PUT: Replacing Resources

PUT requests replace an entire resource with a new version. They are idempotent—sending the same PUT request multiple times has the same effect as sending it once.

**Python:**
```python
import requests
import json

# Updated data
updated_post = {
    "id": 1,
    "title": "Updated Title",
    "body": "This post has been completely replaced.",
    "userId": 1
}

# Set headers
headers = {"Content-Type": "application/json"}

# Make PUT request
response = requests.put(
    "https://jsonplaceholder.typicode.com/posts/1",
    data=json.dumps(updated_post),
    headers=headers
)

# Check if successful
if response.status_code == 200:
    updated_data = response.json()
    print("Resource updated successfully")
else:
    print(f"Error: {response.status_code}")
```

**Java:**
```java
URL url = new URL("https://jsonplaceholder.typicode.com/posts/1");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("PUT");
connection.setRequestProperty("Content-Type", "application/json");
connection.setDoOutput(true);

// Prepare data
String jsonData = "{\"id\":1,\"title\":\"Updated Title\",\"body\":\"This post has been completely replaced.\",\"userId\":1}";

// Send data
try (OutputStream os = connection.getOutputStream()) {
    byte[] input = jsonData.getBytes(StandardCharsets.UTF_8);
    os.write(input, 0, input.length);
}

// Process response
// ...
```

### PATCH: Partial Updates

PATCH requests make partial updates to a resource. Unlike PUT, which replaces the entire resource, PATCH only modifies the specified fields.

**Python:**
```python
import requests
import json

# Only the fields we want to update
patch_data = {
    "title": "Updated Title Only"
}

# Set headers
headers = {"Content-Type": "application/json"}

# Make PATCH request
response = requests.patch(
    "https://jsonplaceholder.typicode.com/posts/1",
    data=json.dumps(patch_data),
    headers=headers
)

# Check if successful
if response.status_code == 200:
    patched_data = response.json()
    print("Resource partially updated")
else:
    print(f"Error: {response.status_code}")
```

**Java:**
```java
URL url = new URL("https://jsonplaceholder.typicode.com/posts/1");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("PATCH");
connection.setRequestProperty("Content-Type", "application/json");
connection.setDoOutput(true);

// Prepare partial data
String jsonData = "{\"title\":\"Updated Title Only\"}";

// Send data
try (OutputStream os = connection.getOutputStream()) {
    byte[] input = jsonData.getBytes(StandardCharsets.UTF_8);
    os.write(input, 0, input.length);
}

// Process response
// ...
```

### DELETE: Removing Resources

DELETE requests remove resources from the server. They are typically idempotent—once a resource is deleted, subsequent DELETE requests for the same resource often return a 404 Not Found.

**Python:**
```python
import requests

# Make DELETE request
response = requests.delete("https://jsonplaceholder.typicode.com/posts/1")

# Check if successful
if response.status_code == 200 or response.status_code == 204:
    print("Resource deleted successfully")
else:
    print(f"Error: {response.status_code}")
    print(response.text)
```

**Java:**
```java
URL url = new URL("https://jsonplaceholder.typicode.com/posts/1");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("DELETE");

int responseCode = connection.getResponseCode();
if (responseCode == 200 || responseCode == 204) {
    System.out.println("Resource deleted successfully");
} else {
    System.out.println("Error: " + responseCode);
    // Read error response
    // ...
}
```

### The Concept of Idempotence

An operation is idempotent if performing it multiple times has the same effect as performing it once. This is a critical concept in API design:

- **Idempotent methods**: GET, PUT, DELETE, HEAD
- **Non-idempotent methods**: POST, PATCH (can be made idempotent with careful design)

Understanding idempotence helps predict API behavior and design reliable systems, especially when retrying failed requests.

### Practice Exercise

1. Use a public API like JSONPlaceholder (https://jsonplaceholder.typicode.com/) to practice each HTTP method.
2. Create a simple client that can perform CRUD operations on a resource.
3. Observe what happens when you:
   - Try to GET a non-existent resource
   - DELETE a resource twice
   - Send an incomplete payload in a POST request
4. Write a function that automatically retries idempotent requests but not non-idempotent ones.

## 5. Response Formats

### JSON: The Standard for Modern APIs

JSON (JavaScript Object Notation) has become the standard format for API responses due to its simplicity, lightweight nature, and native support in JavaScript. It's also easy to work with in virtually all programming languages.

#### Structure of JSON

JSON supports:
- Objects (key-value pairs): `{"name": "John", "age": 30}`
- Arrays: `[1, 2, 3, 4]`
- Strings: `"Hello World"`
- Numbers: `42` or `3.14159`
- Booleans: `true` or `false`
- Null: `null`
- Nested combinations of the above

#### Parsing JSON in Python

```python
import requests
import json

# Make API request
response = requests.get("https://api.openweathermap.org/data/2.5/weather?q=London&appid=your_api_key")

# Parse JSON response
weather_data = response.json()  # requests has built-in JSON parsing

# Or manually:
# weather_data = json.loads(response.text)

# Access nested data
temperature = weather_data["main"]["temp"]
weather_description = weather_data["weather"][0]["description"]

print(f"Current weather in London: {weather_description}, {temperature}°C")

# Converting Python objects to JSON
new_data = {
    "name": "John",
    "languages": ["Python", "Java"],
    "active": True
}

json_string = json.dumps(new_data, indent=2)  # Pretty-printed JSON
print(json_string)
```

#### Parsing JSON in Java

Java requires a library for JSON processing. Common options include Jackson, Gson, and org.json:

```java
// Using org.json
import org.json.JSONObject;
import org.json.JSONArray;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;

// Make API request
URL url = new URL("https://api.openweathermap.org/data/2.5/weather?q=London&appid=your_api_key");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");

// Read response
BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
StringBuilder response = new StringBuilder();
String line;
while ((line = reader.readLine()) != null) {
    response.append(line);
}
reader.close();

// Parse JSON
JSONObject weatherData = new JSONObject(response.toString());
double temperature = weatherData.getJSONObject("main").getDouble("temp");
String weatherDescription = weatherData.getJSONArray("weather").getJSONObject(0).getString("description");

System.out.println("Current weather in London: " + weatherDescription + ", " + temperature + "°C");

// Creating JSON
JSONObject newData = new JSONObject();
newData.put("name", "John");
newData.put("active", true);

JSONArray languages = new JSONArray();
languages.put("Python");
languages.put("Java");
newData.put("languages", languages);

String jsonString = newData.toString(2);  // Pretty-printed JSON
System.out.println(jsonString);
```

### XML: The Legacy Format

XML (eXtensible Markup Language) is an older format still used in some enterprise APIs. While more verbose than JSON, it has strong validation capabilities through DTD and XML Schema.

#### Structure of XML

```xml
<weatherData>
  <location>London</location>
  <temperature unit="celsius">15.2</temperature>
  <conditions>Partly Cloudy</conditions>
  <wind>
    <speed unit="mph">8</speed>
    <direction>NE</direction>
  </wind>
</weatherData>
```

#### Parsing XML in Python

```python
import requests
import xml.etree.ElementTree as ET

# Make API request to an XML API
response = requests.get("https://api.example.com/weather/xml?location=London")

# Parse XML
root = ET.fromstring(response.text)

# Access elements (example paths)
location = root.find("location").text
temperature = root.find("temperature").text
temp_unit = root.find("temperature").get("unit")
wind_speed = root.find("wind/speed").text

print(f"Weather in {location}: {temperature}°{temp_unit}")
```

#### Parsing XML in Java

```java
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import java.io.ByteArrayInputStream;

// Make API request and get XML response
// ...

// Parse XML
DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
DocumentBuilder builder = factory.newDocumentBuilder();
Document document = builder.parse(new ByteArrayInputStream(response.getBytes()));

// Access elements
Element rootElement = document.getDocumentElement();
String location = document.getElementsByTagName("location").item(0).getTextContent();
Element tempElement = (Element) document.getElementsByTagName("temperature").item(0);
String temperature = tempElement.getTextContent();
String tempUnit = tempElement.getAttribute("unit");

System.out.println("Weather in " + location + ": " + temperature + "°" + tempUnit);
```

### Other Response Formats

#### CSV (Comma-Separated Values)

Useful for tabular data:

**Python:**
```python
import requests
import csv
from io import StringIO

response = requests.get("https://api.example.com/data.csv")
csv_data = StringIO(response.text)
reader = csv.DictReader(csv_data)

for row in reader:
    print(f"Country: {row['country']}, Population: {row['population']}")
```

**Java:**
```java
import java.io.BufferedReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

String csvData = response.toString();
BufferedReader reader = new BufferedReader(new StringReader(csvData));

String line;
String[] headers = reader.readLine().split(",");  // Assuming first line has headers

while ((line = reader.readLine()) != null) {
    String[] values = line.split(",");
    System.out.println("Country: " + values[0] + ", Population: " + values[1]);
}
```

#### Binary Data

For files, images, and other non-text data:

**Python:**
```python
import requests

response = requests.get("https://api.example.com/image.jpg", stream=True)
with open("downloaded_image.jpg", "wb") as f:
    for chunk in response.iter_content(chunk_size=8192):
        f.write(chunk)
```

**Java:**
```java
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

URL url = new URL("https://api.example.com/image.jpg");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();

try (InputStream in = connection.getInputStream()) {
    Files.copy(in, Paths.get("downloaded_image.jpg"), StandardCopyOption.REPLACE_EXISTING);
}
```

### Content Negotiation

APIs often support multiple formats. You can request a specific format using the `Accept` header:

**Python:**
```python
import requests

# Request JSON format
headers = {"Accept": "application/json"}
response = requests.get("https://api.example.com/data", headers=headers)

# Request XML format
headers = {"Accept": "application/xml"}
response = requests.get("https://api.example.com/data", headers=headers)
```

**Java:**
```java
URL url = new URL("https://api.example.com/data");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestProperty("Accept", "application/json");
```

### Handling Complex Nested Data

Real-world API responses can have deeply nested structures. Use a methodical approach to explore and extract data:

**Python:**
```python
def extract_value(data, path):
    """
    Extract value from nested JSON using a path string like "main.temp" or "weather[0].description"
    """
    parts = path.split(".")
    current = data
    
    for part in parts:
        # Handle array indexing like "weather[0]"
        if "[" in part and "]" in part:
            key, index_str = part.split("[")
            index = int(index_str.replace("]", ""))
            current = current[key][index]
        else:
            current = current[part]
    
    return current

# Example usage
weather_data = response.json()
temperature = extract_value(weather_data, "main.temp")
description = extract_value(weather_data, "weather[0].description")
```

### Practice Exercise

1. Make requests to an API that supports multiple formats (like JSON and XML).
2. Write functions to parse and extract the same information from both formats.
3. Create a function that can navigate complex nested data structures.
4. Try downloading and saving binary data from an API (like an image).

## 6. Status Codes

### Understanding HTTP Status Codes

HTTP status codes are three-digit numbers that inform clients about the result of their request. They are grouped into five classes:

| Range | Category | Description |
|-------|----------|-------------|
| 1xx | Informational | Request received, continuing process |
| 2xx | Success | Request successfully received, understood, and accepted |
| 3xx | Redirection | Further action needed to complete the request |
| 4xx | Client Error | Request contains bad syntax or cannot be fulfilled |
| 5xx | Server Error | Server failed to fulfill a valid request |

### Common Status Codes and Their Meanings

#### Success Codes (2xx)

- **200 OK**: The request succeeded. The response includes the requested data.
  ```
  GET /users/123 → 200 OK (with user data)
  ```

- **201 Created**: The request succeeded and a new resource was created.
  ```
  POST /users → 201 Created (with the created user data)
  ```

- **204 No Content**: The request succeeded but no content is returned.
  ```
  DELETE /users/123 → 204 No Content
  ```

#### Redirection Codes (3xx)

- **301 Moved Permanently**: The resource has been permanently moved to another location.
  ```
  GET /old-page → 301 Moved Permanently (with Location: /new-page)
  ```

- **304 Not Modified**: The resource hasn't changed since the last request (used with conditional GET).
  ```
  GET /users/123 (with If-None-Match header) → 304 Not Modified
  ```

#### Client Error Codes (4xx)

- **400 Bad Request**: The server cannot process the request due to client error (malformed request, invalid parameters).
  ```
  POST /users (with invalid data) → 400 Bad Request
  ```

- **401 Unauthorized**: Authentication is required and has failed or not been provided.
  ```
  GET /private-resource (without auth) → 401 Unauthorized
  ```

- **403 Forbidden**: The server understood the request but refuses to authorize it.
  ```
  GET /admin-panel (as regular user) → 403 Forbidden
  ```

- **404 Not Found**: The requested resource could not be found.
  ```
  GET /non-existent-page → 404 Not Found
  ```

- **409 Conflict**: The request conflicts with the current state of the server.
  ```
  POST /users (with existing username) → 409 Conflict
  ```

- **429 Too Many Requests**: The user has sent too many requests in a given amount of time (rate limiting).
  ```
  GET /api/data (after exceeding rate limit) → 429 Too Many Requests
  ```

#### Server Error Codes (5xx)

- **500 Internal Server Error**: A generic error message when an unexpected condition was encountered.
  ```
  GET /api/data (when server code fails) → 500 Internal Server Error
  ```

- **502 Bad Gateway**: The server was acting as a gateway or proxy and received an invalid response from the upstream server.
  ```
  GET /api/external-service (when external service is down) → 502 Bad Gateway
  ```

- **503 Service Unavailable**: The server is not ready to handle the request, often due to maintenance or overloading.
  ```
  GET /api/data (during maintenance) → 503 Service Unavailable
  ```

### Checking Status Codes Before Processing Responses

Always check the status code before attempting to process the response:

**Python:**
```python
import requests

response = requests.get("https://api.example.com/resource")

if response.status_code == 200:
    # Process successful response
    data = response.json()
    print(f"Successfully retrieved data: {data}")
elif response.status_code == 404:
    print("Resource not found!")
elif response.status_code == 401:
    print("Authentication required!")
elif 500 <= response.status_code < 600:
    print(f"Server error occurred: {response.status_code}")
else:
    print(f"Unexpected status code: {response.status_code}")
```

**Java:**
```java
URL url = new URL("https://api.example.com/resource");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestMethod("GET");

int statusCode = connection.getResponseCode();

if (statusCode == 200) {
    // Process successful response
    BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
    // Read and process data
} else if (statusCode == 404) {
    System.out.println("Resource not found!");
} else if (statusCode == 401) {
    System.out.println("Authentication required!");
} else if (statusCode >= 500 && statusCode < 600) {
    System.out.println("Server error occurred: " + statusCode);
} else {
    System.out.println("Unexpected status code: " + statusCode);
}
```

### Appropriate Actions for Different Status Codes

Here's how to respond to different status codes programmatically:

| Status Code | Appropriate Action |
|-------------|-------------------|
| 200, 201    | Process the returned data |
| 204         | Consider the operation successful, no data to process |
| 301, 302    | Follow the redirect (most libraries do this automatically) |
| 304         | Use cached data |
| 400         | Fix the request format or parameters |
| 401         | Provide authentication or get a new token |
| 403         | Inform user they don't have permission |
| 404         | Inform user the resource doesn't exist |
| 429         | Wait and retry after the time specified in the Retry-After header |
| 500, 502, 503 | Wait and retry with exponential backoff |

### Status Code Handling Pattern

A pattern for robust status code handling:

**Python:**
```python
import requests
import time

def make_api_request(url, max_retries=3, backoff_factor=1.5):
    """Make an API request with retry logic for certain status codes"""
    retries = 0
    
    while retries < max_retries:
        try:
            response = requests.get(url)
            
            # Success - return the response
            if 200 <= response.status_code < 300:
                return response
                
            # Client errors - don't retry (except 429)
            elif 400 <= response.status_code < 500 and response.status_code != 429:
                print(f"Client error: {response.status_code}")
                return response
                
            # Rate limiting - honor Retry-After header if present
            elif response.status_code == 429:
                retry_after = int(response.headers.get('Retry-After', 5))
                print(f"Rate limited. Waiting {retry_after} seconds.")
                time.sleep(retry_after)
                
            # Server errors and other cases - exponential backoff
            else:
                wait_time = backoff_factor ** retries
                print(f"Received status {response.status_code}. Retrying in {wait_time:.1f} seconds.")
                time.sleep(wait_time)
            
        except requests.exceptions.RequestException as e:
            # Network errors - exponential backoff
            wait_time = backoff_factor ** retries
            print(f"Request failed: {e}. Retrying in {wait_time:.1f} seconds.")
            time.sleep(wait_time)
        
        retries += 1
    
    # If we got here, we ran out of retries
    raise Exception(f"Failed after {max_retries} retries")
```

### Practice Exercise

1. Create a function that makes API requests and handles different status codes appropriately.
2. Test your function against endpoints that might return various status codes:
   - Request a non-existent resource to get a 404
   - Make many requests in short succession to trigger a 429
   - Try accessing a protected resource without authentication for a 401
3. Implement a retry mechanism with exponential backoff for 5xx errors.
4. Create a mock API server that returns different status codes and test your client against it.

## 7. Error Handling

### The Importance of Robust Error Handling

Error handling is critical when working with APIs because many factors are outside your control:
- Network connectivity issues
- API servers going down
- Rate limiting
- Authentication problems
- Invalid input data
- Changes to the API

A well-designed application anticipates these problems and handles them gracefully, providing a better user experience and preventing application crashes.

### Types of Errors to Handle

#### Network Errors

When the API is unreachable or the connection fails:

**Python:**
```python
import requests
import socket

try:
    response = requests.get("https://api.example.com/data", timeout=5)
    response.raise_for_status()  # Raises an exception for 4XX/5XX responses
except requests.exceptions.ConnectionError:
    print("Failed to connect to the server. Check your internet connection.")
except requests.exceptions.Timeout:
    print("The request timed out. The server might be overloaded or down.")
except requests.exceptions.TooManyRedirects:
    print("Too many redirects. The URL may be incorrect.")
except requests.exceptions.HTTPError as err:
    print(f"HTTP error occurred: {err}")
except Exception as err:
    print(f"An unexpected error occurred: {err}")
```

**Java:**
```java
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.io.IOException;

try {
    URL url = new URL("https://api.example.com/data");
    HttpURLConnection connection = (HttpURLConnection) url.openConnection();
    connection.setConnectTimeout(5000);  // 5 seconds
    connection.setReadTimeout(5000);     // 5 seconds
    connection.setRequestMethod("GET");
    
    int responseCode = connection.getResponseCode();
    // Process response...
    
} catch (SocketTimeoutException e) {
    System.out.println("The request timed out. The server might be overloaded or down.");
} catch (UnknownHostException e) {
    System.out.println("Could not find the host. Check the URL and your internet connection.");
} catch (IOException e) {
    System.out.println("An I/O error occurred: " + e.getMessage());
} catch (Exception e) {
    System.out.println("An unexpected error occurred: " + e.getMessage());
}
```

#### Authentication Errors

When credentials are invalid or expired:

**Python:**
```python
def make_authenticated_request(url, api_key):
    try:
        response = requests.get(url, headers={"Authorization": f"Bearer {api_key}"})
        
        if response.status_code == 401:
            # Handle unauthorized error
            print("Authentication failed. Your API key may be invalid or expired.")
            # Potentially refresh the token or prompt for new credentials
            return None
        elif response.status_code == 403:
            print("You don't have permission to access this resource.")
            return None
        
        response.raise_for_status()
        return response.json()
        
    except requests.exceptions.HTTPError as err:
        print(f"HTTP error occurred: {err}")
        return None
```

#### Data Validation Errors

When the API returns an error due to invalid input:

**Python:**
```python
def create_user(api_url, user_data):
    try:
        response = requests.post(api_url, json=user_data)
        
        if response.status_code == 400:
            # Parse validation errors
            errors = response.json().get("errors", {})
            print("Validation errors:")
            for field, messages in errors.items():
                for message in messages:
                    print(f"- {field}: {message}")
            return None
            
        response.raise_for_status()
        return response.json()
        
    except requests.exceptions.HTTPError as err:
        print(f"HTTP error occurred: {err}")
        return None
```

#### Rate Limiting Errors

When you've exceeded the allowed number of requests:

**Python:**
```python
def make_rate_limited_request(url, max_retries=3):
    for attempt in range(max_retries):
        response = requests.get(url)
        
        if response.status_code == 429:
            # Check for Retry-After header
            if "Retry-After" in response.headers:
                wait_seconds = int(response.headers["Retry-After"])
            else:
                wait_seconds = 2 ** attempt  # Exponential backoff
                
            print(f"Rate limit exceeded. Waiting {wait_seconds} seconds...")
            time.sleep(wait_seconds)
            continue
            
        return response
        
    print(f"Failed after {max_retries} retries due to rate limiting")
    return None
```

### Implementing Retry Logic

For transient errors (like network issues or server errors), implementing retry logic with exponential backoff is a best practice:

**Python:**
```python
import requests
import time
import random

def make_request_with_retry(url, max_retries=5, base_delay=1, max_delay=60):
    """Make a request with exponential backoff retry logic"""
    
    for attempt in range(max_retries):
        try:
            response = requests.get(url, timeout=10)
            response.raise_for_status()  # Raise exception for 4xx/5xx status codes
            return response
            
        except (requests.exceptions.RequestException, requests.exceptions.HTTPError) as e:
            # Don't retry for client errors (except for 429 Too Many Requests)
            if hasattr(e, 'response') and 400 <= e.response.status_code < 500 and e.response.status_code != 429:
                print(f"Client error: {e}")
                return e.response
            
            # If we've used all retries, re-raise the exception
            if attempt == max_retries - 1:
                raise
            
            # Calculate delay with exponential backoff and jitter
            delay = min(base_delay * (2 ** attempt) + random.uniform(0, 0.5), max_delay)
            
            print(f"Request failed: {e}. Retrying in {delay:.2f} seconds...")
            time.sleep(delay)
    
    # We shouldn't get here, but just in case
    raise Exception("Retry logic failed")
```

**Java:**
```java
public Response makeRequestWithRetry(String urlString, int maxRetries, double baseDelay, double maxDelay) 
        throws IOException {
    
    Random random = new Random();
    
    for (int attempt = 0; attempt < maxRetries; attempt++) {
        try {
            URL url = new URL(urlString);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setConnectTimeout(10000);
            connection.setReadTimeout(10000);
            
            int statusCode = connection.getResponseCode();
            
            // Success!
            if (statusCode >= 200 && statusCode < 300) {
                return new Response(statusCode, connection.getInputStream());
            }
            
            // Don't retry client errors (except 429)
            if (statusCode >= 400 && statusCode < 500 && statusCode != 429) {
                System.out.println("Client error: " + statusCode);
                return new Response(statusCode, connection.getErrorStream());
            }
            
            // If we've used all retries, return the error
            if (attempt == maxRetries - 1) {
                return new Response(statusCode, connection.getErrorStream());
            }
            
            // Calculate delay with exponential backoff and jitter
            double delay = Math.min(baseDelay * Math.pow(2, attempt) + random.nextDouble() * 0.5, maxDelay);
            
            System.out.println("Request failed with status " + statusCode + ". Retrying in " + 
                    String.format("%.2f", delay) + " seconds...");
            
            Thread.sleep((long)(delay * 1000));
            
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Request interrupted", e);
        }
    }
    
    // We shouldn't get here, but just in case
    throw new IOException("Retry logic failed");
}

// Simple response class to hold status code and data
class Response {
    private int statusCode;
    private InputStream data;
    
    public Response(int statusCode, InputStream data) {
        this.statusCode = statusCode;
        this.data = data;
    }
    
    // Getters...
}
```

### Logging for Troubleshooting

Implementing proper logging is essential for diagnosing API issues:

**Python:**
```python
import logging
import requests

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    filename='api_client.log'
)
logger = logging.getLogger('api_client')

def call_api(url, params=None, headers=None):
    """Call API with logging"""
    try:
        logger.info(f"Making request to {url}")
        
        response = requests.get(url, params=params, headers=headers)
        
        # Log based on response status
        if 200 <= response.status_code < 300:
            logger.info(f"Request succeeded: {response.status_code}")
        else:
            logger.warning(f"Request failed with status: {response.status_code}")
            logger.debug(f"Response body: {response.text[:500]}")  # Log first 500 chars
        
        return response
    
    except requests.exceptions.RequestException as e:
        logger.error(f"Request error: {e}")
        raise
```

### Creating a Robust API Client Class

Putting it all together in a robust API client class:

**Python:**
```python
import requests
import time
import logging
import json
import random

class ApiClient:
    def __init__(self, base_url, api_key=None, timeout=10, max_retries=3):
        self.base_url = base_url
        self.api_key = api_key
        self.timeout = timeout
        self.max_retries = max_retries
        
        # Set up logging
        self.logger = logging.getLogger(__name__)
        
    def _get_headers(self):
        """Get headers for requests"""
        headers = {
            "Content-Type": "application/json",
            "Accept": "application/json"
        }
        
        if self.api_key:
            headers["Authorization"] = f"Bearer {self.api_key}"
            
        return headers
    
    def _make_request(self, method, endpoint, params=None, data=None, retry_on_status=None):
        """
        Make an HTTP request with retry logic
        
        Args:
            method (str): HTTP method (get, post, put, delete)
            endpoint (str): API endpoint (without base URL)
            params (dict, optional): Query parameters
            data (dict, optional): Request body for POST/PUT
            retry_on_status (list, optional): Status codes to retry on, defaults to 5xx
            
        Returns:
            requests.Response: Response object
        """
        if retry_on_status is None:
            retry_on_status = [429, 500, 502, 503, 504]
            
        url = f"{self.base_url}/{endpoint.lstrip('/')}"
        headers = self._get_headers()
        
        self.logger.info(f"Making {method.upper()} request to {url}")
        
        if data:
            self.logger.debug(f"Request data: {json.dumps(data)[:500]}")
        
        for attempt in range(self.max_retries):
            try:
                response = requests.request(
                    method=method,
                    url=url,
                    headers=headers,
                    params=params,
                    json=data if data else None,
                    timeout=self.timeout
                )
                
                # Log response info
                self.logger.info(f"Response status: {response.status_code}")
                
                # Return immediately on success or if not a retryable status code
                if response.status_code < 400 or response.status_code not in retry_on_status:
                    return response
                
                # If we've used all retries, return the response anyway
                if attempt == self.max_retries - 1:
                    return response
                
                # Handle rate limiting
                if response.status_code == 429 and 'Retry-After' in response.headers:
                    sleep_time = int(response.headers['Retry-After'])
                else:
                    # Exponential backoff with jitter
                    sleep_time = (2 ** attempt) + random.uniform(0, 1)
                
                self.logger.warning(
                    f"Request failed with status {response.status_code}. "
                    f"Retrying in {sleep_time:.2f} seconds..."
                )
                
                time.sleep(sleep_time)
                
            except (requests.exceptions.ConnectionError, requests.exceptions.Timeout) as e:
                if attempt == self.max_retries - 1:
                    self.logger.error(f"Request failed after {self.max_retries} attempts: {e}")
                    raise
                
                sleep_time = (2 ** attempt) + random.uniform(0, 1)
                self.logger.warning(f"Request error: {e}. Retrying in {sleep_time:.2f} seconds...")
                time.sleep(sleep_time)
        
        return None  # Should never reach here
    
    # Convenience methods for different HTTP methods
    def get(self, endpoint, params=None):
        return self._make_request("get", endpoint, params=params)
    
    def post(self, endpoint, data=None, params=None):
        return self._make_request("post", endpoint, params=params, data=data)
    
    def put(self, endpoint, data=None, params=None):
        return self._make_request("put", endpoint, params=params, data=data)
    
    def delete(self, endpoint, params=None):
        return self._make_request("delete", endpoint, params=params)
```

### Practice Exercise

1. Implement a comprehensive error handling strategy for an API client.
2. Add appropriate logging to track API calls, responses, and errors.
3. Test your error handling by:
   - Disconnecting from the internet during a request
   - Providing invalid authentication
   - Sending malformed data
   - Simulating rate limiting
4. Extend the `ApiClient` class provided above with more features like:
   - Token refresh functionality
   - Request timeout customization
   - Custom error handling callbacks

## 8. Rate Limiting

### Understanding API Rate Limits

Rate limiting restricts how many requests a client can make to an API within a specific time period. API providers implement rate limits to:
- Prevent abuse and DoS attacks
- Ensure fair usage among all clients
- Maintain service stability
- Create tiered service levels (free vs. paid)

Common rate limit structures include:
- X requests per second
- X requests per minute/hour/day
- Different limits for different endpoints
- Burst limits vs. sustained limits

### How Rate Limits Are Communicated

APIs typically communicate rate limits through HTTP headers:

| Header | Description | Example |
|--------|-------------|---------|
| X-RateLimit-Limit | Maximum requests allowed in period | `X-RateLimit-Limit: 100` |
| X-RateLimit-Remaining | Requests remaining in current period | `X-RateLimit-Remaining: 45` |
| X-RateLimit-Reset | Time when limit resets (Unix timestamp) | `X-RateLimit-Reset: 1612347485` |
| Retry-After | Seconds to wait before retrying | `Retry-After: 30` |

The exact header names may vary between APIs, so check the documentation.

### Detecting Rate Limiting

You can detect rate limiting through:
1. HTTP status code 429 (Too Many Requests)
2. Rate limit headers in the response
3. Error messages in the response body

**Python:**
```python
import requests
import time

def make_request(url):
    response = requests.get(url)
    
    # Check if we're rate limited
    if response.status_code == 429:
        if 'Retry-After' in response.headers:
            retry_after = int(response.headers['Retry-After'])
            print(f"Rate limited. Need to wait {retry_after} seconds.")
            return None, retry_after
        else:
            print("Rate limited. No Retry-After header provided.")
            return None, 60  # Default wait time
    
    # Check remaining limit
    remaining = int(response.headers.get('X-RateLimit-Remaining', 1000))
    reset_time = int(response.headers.get('X-RateLimit-Reset', 0))
    
    current_time = int(time.time())
    time_until_reset = max(0, reset_time - current_time)
    
    print(f"Requests remaining: {remaining}")
    print(f"Rate limit resets in {time_until_reset} seconds")
    
    return response, 0  # No need to wait
```

### Implementing Rate Limit Handling

#### 1. Respecting Retry-After Headers

When rate limited, respect the `Retry-After` header:

**Python:**
```python
def call_api_with_rate_limit_handling(url):
    response = requests.get(url)
    
    if response.status_code == 429:
        if 'Retry-After' in response.headers:
            wait_time = int(response.headers['Retry-After'])
            print(f"Rate limited. Waiting {wait_time} seconds...")
            time.sleep(wait_time)
            # Retry the request
            return call_api_with_rate_limit_handling(url)
        else:
            # No Retry-After header, use default backoff
            print("Rate limited. Using default backoff...")
            time.sleep(30)
            return call_api_with_rate_limit_handling(url)
    
    return response
```

#### 2. Proactive Rate Limiting

Instead of waiting for 429 errors, track rate limits proactively:

**Python:**
```python
import time
import requests

class RateLimitedAPI:
    def __init__(self, base_url, requests_per_minute=60):
        self.base_url = base_url
        self.requests_per_minute = requests_per_minute
        self.request_timestamps = []
    
    def make_request(self, endpoint, method="get", **kwargs):
        """Make a rate-limited request"""
        # Clean up old timestamps
        current_time = time.time()
        self.request_timestamps = [ts for ts in self.request_timestamps 
                                  if current_time - ts < 60]
        
        # Check if we're at the limit
        if len(self.request_timestamps) >= self.requests_per_minute:
            # Calculate time to wait
            oldest_timestamp = min(self.request_timestamps)
            wait_time = 60 - (current_time - oldest_timestamp)
            
            if wait_time > 0:
                print(f"Rate limit reached. Waiting {wait_time:.2f} seconds...")
                time.sleep(wait_time)
        
        # Make the request
        url = f"{self.base_url}/{endpoint.lstrip('/')}"
        response = requests.request(method, url, **kwargs)
        
        # Add timestamp
        self.request_timestamps.append(time.time())
        
        # Handle 429 if our proactive limiting fails
        if response.status_code == 429:
            retry_after = int(response.headers.get('Retry-After', 60))
            print(f"Rate limited anyway. Waiting {retry_after} seconds...")
            time.sleep(retry_after)
            return self.make_request(endpoint, method, **kwargs)
        
        return response
```

#### 3. Token Bucket Implementation

For more sophisticated rate limiting, implement a token bucket algorithm:

**Python:**
```python
import time

class TokenBucket:
    """Token bucket rate limiter"""
    
    def __init__(self, tokens_per_second, max_tokens):
        self.tokens_per_second = tokens_per_second
        self.max_tokens = max_tokens
        self.tokens = max_tokens
        self.last_refill_time = time.time()
    
    def get_token(self):
        """Try to get a token. Returns True if successful, False otherwise."""
        self._refill()
        
        if self.tokens >= 1:
            self.tokens -= 1
            return True
        else:
            return False
    
    def _refill(self):
        """Refill tokens based on elapsed time"""
        now = time.time()
        elapsed = now - self.last_refill_time
        new_tokens = elapsed * self.tokens_per_second
        
        if new_tokens > 0:
            self.tokens = min(self.tokens + new_tokens, self.max_tokens)
            self.last_refill_time = now
    
    def wait_for_token(self):
        """Wait until a token is available and then use it"""
        while not self.get_token():
            # Calculate time until next token
            time_to_wait = (1 - self.tokens) / self.tokens_per_second
            time.sleep(max(0.01, time_to_wait))  # Minimum wait to avoid busy loop
        return True
```

### Handling Multiple Rate Limits

Some APIs have different rate limits for different endpoints. You can implement a system to track multiple limits:

**Python:**
```python
class MultiRateLimiter:
    """Handles multiple rate limits (global and per-endpoint)"""
    
    def __init__(self):
        # Global limiter (e.g., 1000 requests per hour)
        self.global_limiter = TokenBucket(1000/3600, 1000)
        
        # Per-endpoint limiters
        self.endpoint_limiters = {
            "search": TokenBucket(30/60, 30),  # 30 per minute
            "users": TokenBucket(300/60, 300),  # 300 per minute
            # Add more endpoints as needed
        }
    
    def wait_for_request(self, endpoint):
        """Wait until request can be made for this endpoint"""
        # First check global limit
        self.global_limiter.wait_for_token()
        
        # Then check endpoint-specific limit if it exists
        if endpoint in self.endpoint_limiters:
            self.endpoint_limiters[endpoint].wait_for_token()
```

### Practice Exercise

1. Create a rate-limited API client that respects the rate limits of a public API.
2. Implement the token bucket algorithm and use it to limit your requests.
3. Write code to parse and utilize rate limit headers from responses.
4. Test your implementation by making many requests and observing how your client throttles itself to avoid 429 errors.
5. Create a visualization (console output or graph) showing your request rate over time.

## 9. Versioning

### Why API Versioning Matters

API versioning allows providers to evolve their APIs without breaking existing client applications. Without versioning, any change to an API could potentially break all clients using it. Versioning gives both API providers and consumers a smooth transition path when changes are necessary.

Key benefits of versioning:
- Allows introduction of new features
- Enables deprecating or removing outdated functionality
- Provides backward compatibility for existing clients
- Allows major architectural changes over time
- Helps with documentation and support

### Common Versioning Strategies

#### 1. URI Path Versioning

Including the version in the URL path:

```
https://api.example.com/v1/users
https://api.example.com/v2/users
```

**Python:**
```python
# Client for v1
v1_client = ApiClient("https://api.example.com/v1")
response = v1_client.get("users")

# Client for v2
v2_client = ApiClient("https://api.example.com/v2")
response = v2_client.get("users")
```

**Java:**
```java
// Client for v1
ApiClient v1Client = new ApiClient("https://api.example.com/v1");
Response v1Response = v1Client.get("users");

// Client for v2
ApiClient v2Client = new ApiClient("https://api.example.com/v2");
Response v2Response = v2Client.get("users");
```

#### 2. Query Parameter Versioning

Including the version as a query parameter:

```
https://api.example.com/users?version=1
https://api.example.com/users?version=2
```

**Python:**
```python
api_client = ApiClient("https://api.example.com")

# Get v1 data
response_v1 = api_client.get("users", params={"version": "1"})

# Get v2 data
response_v2 = api_client.get("users", params={"version": "2"})
```

#### 3. HTTP Header Versioning

Using a custom HTTP header to specify the version:

```
Accept: application/vnd.example.v1+json
Accept: application/vnd.example.v2+json
```

**Python:**
```python
import requests

# Get v1 data
headers_v1 = {"Accept": "application/vnd.example.v1+json"}
response_v1 = requests.get("https://api.example.com/users", headers=headers_v1)

# Get v2 data
headers_v2 = {"Accept": "application/vnd.example.v2+json"}
response_v2 = requests.get("https://api.example.com/users", headers=headers_v2)
```

**Java:**
```java
// Get v1 data
URL url = new URL("https://api.example.com/users");
HttpURLConnection connection = (HttpURLConnection) url.openConnection();
connection.setRequestProperty("Accept", "application/vnd.example.v1+json");
```

#### 4. Content Type Versioning

Embedding the version in the content type:

```
Content-Type: application/json; version=1
Content-Type: application/json; version=2
```

### Handling API Version Changes

When an API you're using releases a new version, follow these steps:

1. **Read the Changelog**: Understand what's changed and why.
2. **Review Deprecated Features**: Identify any methods or parameters you're using that will be deprecated.
3. **Test With Both Versions**: Test your application against both the old and new versions in a development environment.
4. **Update Your Code**: Make necessary changes to support the new version.
5. **Implement Fallback Logic**: Consider having fallback code that can work with multiple versions.

**Python Example with Version Fallback:**
```python
def get_user_data(user_id):
    """Get user data with fallback to older API version if needed"""
    # Try latest version first
    try:
        headers = {"Accept": "application/vnd.example.v2+json"}
        response = requests.get(f"https://api.example.com/users/{user_id}", headers=headers)
        
        # If successful, parse v2 response format
        if response.status_code == 200:
            data = response.json()
            return {
                "id": data["id"],
                "name": data["display_name"],  # v2 uses display_name
                "email": data["email_address"]  # v2 uses email_address
            }
    except Exception as e:
        print(f"Error with v2 API: {e}")
    
    # Fallback to v1
    try:
        headers = {"Accept": "application/vnd.example.v1+json"}
        response = requests.get(f"https://api.example.com/users/{user_id}", headers=headers)
        
        # Parse v1 response format
        if response.status_code == 200:
            data = response.json()
            return {
                "id": data["id"],
                "name": data["name"],  # v1 uses name
                "email": data["email"]  # v1 uses email
            }
    except Exception as e:
        print(f"Error with v1 API: {e}")
    
    return None  # Both versions failed
```

### Versioning Best Practices

1. **Semantic Versioning**: Follow the MAJOR.MINOR.PATCH pattern where:
   - MAJOR: Breaking changes
   - MINOR: New features, backwards-compatible
   - PATCH: Bug fixes, backwards-compatible

2. **Maintain Multiple Versions**: Keep older versions running during transition periods.

3. **Clear Deprecation Policy**: Communicate when old versions will be discontinued.

4. **Version-Specific Documentation**: Maintain separate documentation for each version.

5. **Version in Responses**: Include version information in API responses for debugging.

### Practice Exercise

1. Find a public API that supports multiple versions (GitHub, Twitter, etc.).
2. Write a client that can work with two different versions of the API.
3. Create a function that automatically detects which version of an API is available and adapts accordingly.
4. Design your own simple API and create a version upgrade strategy, including what would change between versions.

## 10. Documentation and Testing

### Understanding API Documentation

Good API documentation is essential for developers to effectively use an API. It typically includes:

1. **Getting Started Guide**: Basic information on authentication and making your first request
2. **Reference Documentation**: Details of each endpoint, including:
   - URL structure
   - HTTP method
   - Required and optional parameters
   - Request and response formats
   - Example requests and responses
   - Error codes and messages
3. **Tutorials and Use Cases**: Common scenarios and how to implement them
4. **SDKs and Client Libraries**: Official libraries for different programming languages
5. **Change Log**: History of API changes and version differences

### How to Read API Documentation

Reading API documentation effectively is a skill:

1. **Start with the overview**: Understand the general structure and concepts.
2. **Look for authentication details**: Figure out how to authenticate your requests.
3. **Identify the endpoints you need**: Find specific functionality you want to use.
4. **Check the request format**: Understand required and optional parameters.
5. **Examine response examples**: Know what data to expect back.
6. **Look for rate limits**: Understand usage restrictions.
7. **Find error handling information**: Learn how to handle failures.

### API Documentation Example

Here's how a typical API documentation entry might look:

```
GET /users/{user_id}

Retrieves details for a specific user.

Path Parameters:
- user_id (required): The ID of the user to retrieve

Query Parameters:
- include_inactive (optional): Set to 'true' to include inactive users. Default: false
- fields (optional): Comma-separated list of fields to include in the response

Request Headers:
- Authorization (required): Bearer {access_token}
- Accept (optional): application/json (default) or application/xml

Response:
- 200 OK: User details retrieved successfully
- 404 Not Found: User does not exist
- 401 Unauthorized: Invalid or missing authentication

Example Request:
GET /users/12345?fields=name,email HTTP/1.1
Host: api.example.com
Authorization: Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...

Example Response (200 OK):
{
  "id": "12345",
  "name": "John Doe",
  "email": "john.doe@example.com"
}
```

### Testing APIs

#### Manual Testing with Tools

Several tools can help you test APIs manually:

1. **Postman**: A graphical interface for building and testing HTTP requests
2. **curl**: Command-line tool for making HTTP requests
3. **httpie**: A more user-friendly command-line HTTP client

**curl Example:**
```bash
# Basic GET request
curl https://api.example.com/users

# GET request with authentication
curl -H "Authorization: Bearer YOUR_TOKEN" https://api.example.com/users

# POST request with JSON data
curl -X POST \
  -H "Content-Type: application/json" \
  -d '{"name": "John Doe", "email": "john@example.com"}' \
  https://api.example.com/users
```

#### Automated API Testing

Writing automated tests for API interactions ensures reliability:

**Python with pytest:**
```python
import pytest
import requests

# API credentials
API_KEY = "your_api_key"
BASE_URL = "https://api.example.com"

def get_auth_headers():
    return {"Authorization": f"Bearer {API_KEY}"}

def test_get_user():
    """Test retrieving a user"""
    user_id = "12345"
    response = requests.get(f"{BASE_URL}/users/{user_id}", headers=get_auth_headers())
    
    # Check status code
    assert response.status_code == 200
    
    # Check response structure
    data = response.json()
    assert "id" in data
    assert "name" in data
    assert "email" in data
    
    # Check specific data
    assert data["id"] == user_id
    assert "@" in data["email"]  # Basic email validation

def test_create_user():
    """Test creating a user"""
    new_user = {
        "name": "Test User",
        "email": "test@example.com",
        "role": "user"
    }
    
    response = requests.post(
        f"{BASE_URL}/users",
        headers={**get_auth_headers(), "Content-Type": "application/json"},
        json=new_user
    )
    
    # Check status code for successful creation
    assert response.status_code == 201
    
    # Verify the created user has an ID
    data = response.json()
    assert "id" in data
    
    # Clean up - delete the test user
    user_id = data["id"]
    delete_response = requests.delete(f"{BASE_URL}/users/{user_id}", headers=get_auth_headers())
    assert delete_response.status_code in [200, 204]
```

**Java with JUnit and RestAssured:**
```java
import io.restassured.RestAssured;
import io.restassured.http.ContentType;
import org.junit.BeforeClass;
import org.junit.Test;
import static io.restassured.RestAssured.*;
import static org.hamcrest.Matchers.*;

public class ApiTests {
    
    private static final String API_KEY = "your_api_key";
    
    @BeforeClass
    public static void setup() {
        RestAssured.baseURI = "https://api.example.com";
    }
    
    @Test
    public void testGetUser() {
        String userId = "12345";
        
        given()
            .header("Authorization", "Bearer " + API_KEY)
        .when()
            .get("/users/" + userId)
        .then()
            .statusCode(200)
            .contentType(ContentType.JSON)
            .body("id", equalTo(userId))
            .body("name", notNullValue())
            .body("email", containsString("@"));
    }
    
    @Test
    public void testCreateUser() {
        String newUserId = 
        given()
            .header("Authorization", "Bearer " + API_KEY)
            .contentType(ContentType.JSON)
            .body("{"
                + "\"name\": \"Test User\","
                + "\"email\": \"test@example.com\","
                + "\"role\": \"user\""
                + "}")
        .when()
            .post("/users")
        .then()
            .statusCode(201)
            .contentType(ContentType.JSON)
            .body("id", notNullValue())
            .extract().path("id");
        
        // Clean up - delete the test user
        given()
            .header("Authorization", "Bearer " + API_KEY)
        .when()
            .delete("/users/" + newUserId)
        .then()
            .statusCode(anyOf(is(200), is(204)));
    }
}
```

### Creating Mock APIs for Testing

When developing against an API that's not ready or when you want to test edge cases, mock APIs are useful:

**Python with Flask:**
```python
from flask import Flask, jsonify, request

app = Flask(__name__)

# In-memory database
users = {
    "12345": {
        "id": "12345",
        "name": "John Doe",
        "email": "john@example.com"
    }
}

@app.route('/users/<user_id>', methods=['GET'])
def get_user(user_id):
    # Simulate authentication check
    auth_header = request.headers.get('Authorization')
    if not auth_header or not auth_header.startswith('Bearer '):
        return jsonify({"error": "Unauthorized"}), 401
    
    # Check if user exists
    if user_id not in users:
        return jsonify({"error": "User not found"}), 404
    
    # Return user data
    return jsonify(users[user_id])

@app.route('/users', methods=['POST'])
def create_user():
    # Simulate authentication check
    auth_header = request.headers.get('Authorization')
    if not auth_header or not auth_header.startswith('Bearer '):
        return jsonify({"error": "Unauthorized"}), 401
    
    # Get request data
    data = request.json
    
    # Validate required fields
    if not data or 'name' not in data or 'email' not in data:
        return jsonify({"error": "Missing required fields"}), 400
    
    # Create user ID (normally would be generated by database)
    import uuid
    user_id = str(uuid.uuid4())
    
    # Store user
    users[user_id] = {
        "id": user_id,
        "name": data['name'],
        "email": data['email'],
        "role": data.get('role', 'user')  # Default role
    }
    
    return jsonify(users[user_id]), 201

if __name__ == '__main__':
    app.run(debug=True)
```

### Documentation Tools

Several tools can help you create and maintain API documentation:

1. **Swagger/OpenAPI**: Define your API structure in a standard format that can generate documentation, client libraries, and more.
2. **Postman Documentation**: Create documentation directly from your Postman collections.
3. **API Blueprint**: A markdown-based documentation format.
4. **Docusaurus**: A documentation website generator popular for API docs.

### Practice Exercise

1. Find a public API with good documentation (GitHub, Stripe, Twilio, etc.) and study its structure.
2. Use a tool like Postman or curl to make test requests to a public API.
3. Write automated tests for basic CRUD operations against a public API or your mock API.
4. Create a simple mock API for testing using Flask, Express.js, or another web framework.
5. Document a small API you've created using OpenAPI/Swagger.

## 11. API Security Best Practices

### Common API Security Vulnerabilities

APIs can be vulnerable to various security threats:

1. **Authentication Weaknesses**: Poor token management, weak password policies
2. **Authorization Issues**: Missing permission checks, horizontal privilege escalation
3. **Data Exposure**: Revealing sensitive data in responses
4. **Injection Attacks**: SQL injection, command injection
5. **Rate Limiting Bypass**: Allowing too many requests, leading to DoS
6. **Man-in-the-Middle**: Intercepting unencrypted communications
7. **Insecure Direct Object References**: Allowing access to unauthorized resources

### Security Implementation Best Practices

#### 1. Always Use HTTPS

Encrypt all API traffic using TLS:

**Python:**
```python
import requests

# Always use HTTPS URLs
response = requests.get("https://api.example.com/data")

# Verify SSL certificates (enabled by default in requests)
response = requests.get("https://api.example.com/data", verify=True)

# You can also specify a certificate bundle
response = requests.get("https://api.example.com/data", verify="/path/to/certfile")
```

**Java:**
```java
import javax.net.ssl.HttpsURLConnection;
import java.net.URL;

URL url = new URL("https://api.example.com/data");
HttpsURLConnection connection = (HttpsURLConnection) url.openConnection();

// Enable hostname verification (default is true)
connection.setHostnameVerifier(HttpsURLConnection.getDefaultHostnameVerifier());
```

#### 2. Implement Proper Authentication

Use secure authentication methods:

**OAuth 2.0 Flow in Python:**
```python
import requests

# Step 1: Authorization Request (redirect user to this URL)
auth_url = "https://auth.example.com/oauth/authorize"
auth_params = {
    "response_type": "code",
    "client_id": "YOUR_CLIENT_ID",
    "redirect_uri": "YOUR_REDIRECT_URI",
    "scope": "read write",
    "state": "RANDOM_STATE_STRING"  # Prevent CSRF
}

# After user authorizes, they are redirected to your redirect_uri with a code

# Step 2: Exchange code for token
token_url = "https://auth.example.com/oauth/token"
token_params = {
    "grant_type": "authorization_code",
    "code": "AUTH_CODE_FROM_REDIRECT",
    "redirect_uri": "YOUR_REDIRECT_URI",
    "client_id": "YOUR_CLIENT_ID",
    "client_secret": "YOUR_CLIENT_SECRET"
}

token_response = requests.post(token_url, data=token_params)
tokens = token_response.json()
access_token = tokens["access_token"]

# Step 3: Use token in API requests
headers = {"Authorization": f"Bearer {access_token}"}
api_response = requests.get("https://api.example.com/data", headers=headers)
```

#### 3. Secure Storage of Credentials

Never hardcode or expose credentials:

**Python:**
```python
import os
from dotenv import load_dotenv

# Load credentials from environment variables
load_dotenv()  # Load variables from .env file

api_key = os.environ.get("API_KEY")
client_secret = os.environ.get("CLIENT_SECRET")

# Use credentials in requests
headers = {"Authorization": f"Bearer {api_key}"}
```

**Java:**
```java
// Load from environment variables
String apiKey = System.getenv("API_KEY");
String clientSecret = System.getenv("CLIENT_SECRET");

// Or from properties file (not included in version control)
Properties prop = new Properties();
try (FileInputStream input = new FileInputStream("config.properties")) {
    prop.load(input);
}
String apiKey = prop.getProperty("api.key");
```

#### 4. Input Validation

Always validate and sanitize input:

**Python:**
```python
def validate_user_input(user_data):
    errors = {}
    
    # Check required fields
    if "email" not in user_data or not user_data["email"]:
        errors["email"] = "Email is required"
    elif not re.match(r"[^@]+@[^@]+\.[^@]+", user_data["email"]):
        errors["email"] = "Invalid email format"
    
    # Validate numeric values
    if "age" in user_data:
        try:
            age = int(user_data["age"])
            if age < 0 or age > 120:
                errors["age"] = "Age must be between 0 and 120"
        except ValueError:
            errors["age"] = "Age must be a number"
    
    # Sanitize text fields
    if "name" in user_data:
        # Remove any HTML tags
        user_data["name"] = re.sub(r"<[^>]*>", "", user_data["name"])
    
    return errors, user_data
```

#### 5. Protect Against Common Attacks

Guard against injection and other attacks:

**SQL Injection Prevention (Python with SQLAlchemy):**
```python
from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker

engine = create_engine("postgresql://user:pass@localhost/dbname")
Session = sessionmaker(bind=engine)
session = Session()

# UNSAFE:
# user_id = request.args.get("user_id")
# query = f"SELECT * FROM users WHERE id = {user_id}"  # VULNERABLE!
# result = session.execute(query)

# SAFE:
user_id = request.args.get("user_id")
# Use parameterized queries
result = session.execute(
    text("SELECT * FROM users WHERE id = :user_id"),
    {"user_id": user_id}
)
```

**XSS Prevention:**
```python
import html

def render_user_content(content):
    # Escape HTML special characters
    safe_content = html.escape(content)
    return safe_content
```

#### 6. Implement Proper Logging

Log security events without exposing sensitive data:

**Python:**
```python
import logging
import re

# Configure logging
logging.basicConfig(
    filename="api_security.log",
    level=logging.INFO,
    format="%(asctime)s - %(levelname)s - %(message)s"
)

def log_api_request(request, user_id=None):
    # Mask sensitive data
    headers = request.headers.copy()
    if "Authorization" in headers:
        headers["Authorization"] = "Bearer [REDACTED]"
    
    # Log request details
    logging.info({
        "method": request.method,
        "path": request.path,
        "user_id": user_id,
        "ip": request.remote_addr,
        "user_agent": request.user_agent.string,
        "headers": headers
    })

def log_authentication_failure(username, ip, reason):
    logging.warning(f"Auth failure: {reason}, User: {mask_pii(username)}, IP: {ip}")

def mask_pii(text):
    """Mask personally identifiable information"""
    # Mask email addresses
    text = re.sub(r"([a-zA-Z0-9._%+-]+)@([a-zA-Z0-9.-]+\.[a-zA-Z]{2,})", r"***@\2", text)
    
    # Mask credit card numbers
    text = re.sub(r"\b(\d{4})\d{8,12}(\d{4})\b", r"\1********\2", text)
    
    return text
```

### Practice Exercise

1. Audit an existing API client for security vulnerabilities:
   - Check for hardcoded credentials
   - Verify HTTPS usage
   - Look for proper input validation
   - Check token handling
2. Implement secure credential storage using environment variables or a secure configuration system.
3. Create a function to validate and sanitize API inputs before sending them.
4. Implement proper token handling with secure storage and automatic refresh for expired tokens.
5. Write a logging system that captures security events without exposing sensitive data.

## 12. Real-World API Integration Examples

Let's look at some real-world examples of integrating with popular APIs:

### Example 1: Weather Forecast Application

**Requirements:**
- Display current weather and 5-day forecast for a location
- Show temperature, humidity, wind speed, and conditions
- Allow searching by city name or coordinates

**API Choice:** OpenWeatherMap API

**Implementation in Python:**

```python
import requests
import os
from datetime import datetime

class WeatherApp:
    def __init__(self, api_key=None):
        self.api_key = api_key or os.environ.get("OPENWEATHERMAP_API_KEY")
        self.base_url = "https://api.openweathermap.org/data/2.5"
        
    def get_current_weather(self, city=None, lat=None, lon=None):
        """Get current weather for a location"""
        # Prepare parameters
        params = {
            "appid": self.api_key,
            "units": "metric"  # Use metric units (Celsius)
        }
        
        # Set location parameter
        if city:
            params["q"] = city
        elif lat is not None and lon is not None:
            params["lat"] = lat
            params["lon"] = lon
        else:
            raise ValueError("Either city name or coordinates must be provided")
        
        # Make request
        response = requests.get(f"{self.base_url}/weather", params=params)
        
        # Check for errors
        if response.status_code != 200:
            return self._handle_error(response)
        
        # Parse and format response
        data = response.json()
        return {
            "location": data["name"],
            "country": data["sys"]["country"],
            "temperature": round(data["main"]["temp"]),
            "feels_like": round(data["main"]["feels_like"]),
            "humidity": data["main"]["humidity"],
            "wind_speed": data["wind"]["speed"],
            "conditions": data["weather"][0]["description"],
            "icon": data["weather"][0]["icon"],
            "timestamp": datetime.fromtimestamp(data["dt"]).strftime("%Y-%m-%d %H:%M:%S")
        }
    
    def get_forecast(self, city=None, lat=None, lon=None):
        """Get 5-day forecast for a location"""
        # Prepare parameters
        params = {
            "appid": self.api_key,
            "units": "metric"
        }
        
        # Set location parameter
        if city:
            params["q"] = city
        elif lat is not None and lon is not None:
            params["lat"] = lat
            params["lon"] = lon
        else:
            raise ValueError("Either city name or coordinates must be provided")
        
        # Make request
        response = requests.get(f"{self.base_url}/forecast", params=params)
        
        # Check for errors
        if response.status_code != 200:
            return self._handle_error(response)
        
        # Parse and format response
        data = response.json()
        forecasts = []
        
        # Group forecasts by day (OpenWeatherMap returns 3-hour intervals)
        days = {}
        for item in data["list"]:
            # Get date (without time)
            date = datetime.fromtimestamp(item["dt"]).strftime("%Y-%m-%d")
            
            if date not in days:
                days[date] = []
            
            days[date].append({
                "timestamp": item["dt"],
                "temperature": round(item["main"]["temp"]),
                "conditions": item["weather"][0]["description"],
                "icon": item["weather"][0]["icon"],
                "wind_speed": item["wind"]["speed"],
                "humidity": item["main"]["humidity"]
            })
        
        # Format each day's forecast (using midday forecast as representative)
        for date, items in days.items():
            # Get the forecast closest to midday
            midday_forecast = min(items, key=lambda x: abs(
                datetime.fromtimestamp(x["timestamp"]).hour - 12
            ))
            
            # Format the day's forecast
            readable_date = datetime.strptime(date, "%Y-%m-%d").strftime("%A, %b %d")
            forecasts.append({
                "date": readable_date,
                "temperature": midday_forecast["temperature"],
                "conditions": midday_forecast["conditions"],
                "icon": midday_forecast["icon"],
                "wind_speed": midday_forecast["wind_speed"],
                "humidity": midday_forecast["humidity"]
            })
        
        return {
            "location": data["city"]["name"],
            "country": data["city"]["country"],
            "forecasts": forecasts[:5]  # Limit to 5 days
        }
    
    def _handle_error(self, response):
        """Handle API error responses"""
        try:
            error_data = response.json()
            error_message = error_data.get("message", "Unknown error")
        except:
            error_message = f"Error: HTTP {response.status_code}"
        
        return {
            "error": True,
            "message": error_message,
            "status_code": response.status_code
        }

# Example usage
if __name__ == "__main__":
    weather_app = WeatherApp()
    
    # Get current weather
    current = weather_app.get_current_weather(city="London")
    if "error" not in current:
        print(f"Current weather in {current['location']}:")
        print(f"Temperature: {current['temperature']}°C")
        print(f"Conditions: {current['conditions']}")
        print(f"Wind Speed: {current['wind_speed']} m/s")
        print(f"Humidity: {current['humidity']}%")
    else:
        print(f"Error: {current['message']}")
    
    # Get forecast
    forecast = weather_app.get_forecast(city="London")
    if "error" not in forecast:
        print(f"\n5-day forecast for {forecast['location']}:")
        for day in forecast["forecasts"]:
            print(f"{day['date']}: {day['temperature']}°C, {day['conditions']}")
```

### Example 2: GitHub Repository Browser

**Requirements:**
- List a user's repositories
- Show repository details (stars, forks, language)
- Display recent commits

**API Choice:** GitHub REST API

**Implementation in Java:**

```java
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONObject;

public class GitHubApp {
    private final String apiToken;
    private final String baseUrl = "https://api.github.com";
    
    public GitHubApp(String apiToken) {
        this.apiToken = apiToken;
    }
    
    public List<Repository> getUserRepositories(String username) throws IOException {
        String endpoint = "/users/" + username + "/repos";
        JSONArray reposJson = makeRequest(endpoint);
        
        List<Repository> repositories = new ArrayList<>();
        for (int i = 0; i < reposJson.length(); i++) {
            JSONObject repoJson = reposJson.getJSONObject(i);
            
            Repository repo = new Repository();
            repo.id = repoJson.getInt("id");
            repo.name = repoJson.getString("name");
            repo.fullName = repoJson.getString("full_name");
            repo.description = repoJson.isNull("description") ? "" : repoJson.getString("description");
            repo.url = repoJson.getString("html_url");
            repo.stars = repoJson.getInt("stargazers_count");
            repo.forks = repoJson.getInt("forks_count");
            repo.language = repoJson.isNull("language") ? "Unknown" : repoJson.getString("language");
            repo.createdAt = parseDate(repoJson.getString("created_at"));
            
            repositories.add(repo);
        }
        
        return repositories;
    }
    
    public List<Commit> getRepositoryCommits(String owner, String repo, int limit) throws IOException {
        String endpoint = "/repos/" + owner + "/" + repo + "/commits";
        JSONArray commitsJson = makeRequest(endpoint);
        
        List<Commit> commits = new ArrayList<>();
        int count = Math.min(commitsJson.length(), limit);
        
        for (int i = 0; i < count; i++) {
            JSONObject commitJson = commitsJson.getJSONObject(i);
            
            Commit commit = new Commit();
            commit.sha = commitJson.getString("sha");
            
            JSONObject commitDetails = commitJson.getJSONObject("commit");
            commit.message = commitDetails.getString("message");
            
            JSONObject authorJson = commitDetails.getJSONObject("author");
            commit.authorName = authorJson.getString("name");
            commit.authorEmail = authorJson.getString("email");
            commit.date = parseDate(authorJson.getString("date"));
            
            commits.add(commit);
        }
        
        return commits;
    }
    
    private JSONArray makeRequest(String endpoint) throws IOException {
        URL url = new URL(baseUrl + endpoint);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        
        // Set headers
        connection.setRequestProperty("Accept", "application/vnd.github.v3+json");
        if (apiToken != null && !apiToken.isEmpty()) {
            connection.setRequestProperty("Authorization", "token " + apiToken);
        }
        
        // Check response code
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            throw new IOException("API request failed with status: " + responseCode);
        }
        
        // Read response
        BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
        StringBuilder response = new StringBuilder();
        String line;
        
        while ((line = reader.readLine()) != null) {
            response.append(line);
        }
        reader.close();
        
        // Parse JSON response
        return new JSONArray(response.toString());
    }
    
    private Date parseDate(String dateString) {
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
            return formatter.parse(dateString);
        } catch (Exception e) {
            return new Date();  // Return current date as fallback
        }
    }
    
    // Data models
    public static class Repository {
        public int id;
        public String name;
        public String fullName;
        public String description;
        public String url;
        public int stars;
        public int forks;
        public String language;
        public Date createdAt;
        
        @Override
        public String toString() {
            SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd, yyyy");
            return String.format("%s - %d★ %d🍴 (%s) - Created on %s",
                    fullName, stars, forks, language, dateFormat.format(createdAt));
        }
    }
    
    public static class Commit {
        public String sha;
        public String message;
        public String authorName;
        public String authorEmail;
        public Date date;
        
        @Override
        public String toString() {
            SimpleDateFormat dateFormat = new SimpleDateFormat("MMM dd, yyyy HH:mm");
            return String.format("[%s] %s <%s> - %s",
                    dateFormat.format(date), authorName, authorEmail, message);
        }
    }
    
    // Example usage
    public static void main(String[] args) {
        try {
            String token = System.getenv("GITHUB_TOKEN");  // Get from environment variable
            GitHubApp app = new GitHubApp(token);
            
            // Get repositories for a user
            String username = "octocat";
            List<Repository> repos = app.getUserRepositories(username);
            
            System.out.println("Repositories for " + username + ":");
            for (Repository repo : repos) {
                System.out.println(repo);
            }
            
            // Get commits for a repository
            if (!repos.isEmpty()) {
                Repository firstRepo = repos.get(0);
                String[] parts = firstRepo.fullName.split("/");
                List<Commit> commits = app.getRepositoryCommits(parts[0], parts[1], 5);
                
                System.out.println("\nRecent commits for " + firstRepo.fullName + ":");
                for (Commit commit : commits) {
                    System.out.println(commit);
                }
            }
            
        } catch (IOException e) {
            System.err.println("Error: " + e.getMessage());
        }
    }
}
```

### Example 3: E-commerce Product Inventory System

**Requirements:**
- Retrieve product catalog from an API
- Add, update, and remove products
- Handle product categories and attributes

**API Choice:** Custom REST API

**Implementation in Python:**

```python
import requests
import json
import logging
import time
import os
from typing import Dict, List, Optional, Any, Union

# Set up logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    filename='product_api.log'
)
logger = logging.getLogger('product_api')

class ProductAPI:
    def __init__(self, api_url: str, api_key: str = None):
        """Initialize the Product API client"""
        self.api_url = api_url.rstrip('/')
        self.api_key = api_key or os.environ.get('PRODUCT_API_KEY')
        self.session = requests.Session()
        
        # Set default headers
        self.session.headers.update({
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-API-Key': self.api_key
        })
    
    def get_products(self, category: str = None, page: int = 1, 
                     limit: int = 50, sort_by: str = 'name') -> Dict:
        """Get list of products with optional filtering"""
        endpoint = '/products'
        params = {
            'page': page,
            'limit': limit,
            'sort': sort_by
        }
        
        if category:
            params['category'] = category
        
        return self._make_request('GET', endpoint, params=params)
    
    def get_product(self, product_id: str) -> Dict:
        """Get details for a specific product"""
        endpoint = f'/products/{product_id}'
        return self._make_request('GET', endpoint)
    
    def create_product(self, product_data: Dict) -> Dict:
        """Create a new product"""
        self._validate_product_data(product_data, is_new=True)
        endpoint = '/products'
        return self._make_request('POST', endpoint, json=product_data)
    
    def update_product(self, product_id: str, product_data: Dict) -> Dict:
        """Update an existing product"""
        self._validate_product_data(product_data, is_new=False)
        endpoint = f'/products/{product_id}'
        return self._make_request('PUT', endpoint, json=product_data)
    
    def delete_product(self, product_id: str) -> Dict:
        """Delete a product"""
        endpoint = f'/products/{product_id}'
        return self._make_request('DELETE', endpoint)
    
    def get_categories(self) -> List[Dict]:
        """Get list of product categories"""
        endpoint = '/categories'
        return self._make_request('GET', endpoint)
    
    def search_products(self, query: str, category: str = None, 
                       page: int = 1, limit: int = 20) -> Dict:
        """Search for products"""
        endpoint = '/products/search'
        params = {
            'q': query,
            'page': page,
            'limit': limit
        }
        
        if category:
            params['category'] = category
        
        return self._make_request('GET', endpoint, params=params)
    
    def bulk_update_prices(self, price_updates: List[Dict]) -> Dict:
        """Update prices for multiple products at once"""
        endpoint = '/products/price-update'
        return self._make_request('POST', endpoint, json={'updates': price_updates})
    
    def _validate_product_data(self, data: Dict, is_new: bool = False) -> None:
        """Validate product data before sending to API"""
        required_fields = ['name', 'price', 'category_id', 'description', 'sku']
        
        if is_new:  # Only check required fields for new products
            missing = [field for field in required_fields if field not in data]
            if missing:
                raise ValueError(f"Missing required fields: {', '.join(missing)}")
        
        # Validate price format
        if 'price' in data and not isinstance(data['price'], (int, float)):
            raise ValueError("Price must be a number")
        
        # Validate SKU format if present
        if 'sku' in data and not isinstance(data['sku'], str):
            raise ValueError("SKU must be a string")
    
    def _make_request(self, method: str, endpoint: str, 
                     params: Dict = None, json: Dict = None, 
                     retry_count: int = 3) -> Union[Dict, List]:
        """Make HTTP request to the API with retry logic"""
        url = f"{self.api_url}{endpoint}"
        logger.info(f"Making {method} request to {url}")
        
        for attempt in range(retry_count):
            try:
                response = self.session.request(
                    method=method,
                    url=url,
                    params=params,
                    json=json,
                    timeout=10
                )
                
                # Log response status
                logger.info(f"Response status: {response.status_code}")
                
                # Handle different status codes
                if 200 <= response.status_code < 300:
                    return response.json()
                
                elif response.status_code == 429:  # Rate limited
                    retry_after = int(response.headers.get('Retry-After', 5))
                    logger.warning(f"Rate limited. Waiting {retry_after} seconds.")
                    time.sleep(retry_after)
                    continue
                    
                elif response.status_code == 401:
                    logger.error("Authentication failed. Check your API key.")
                    raise AuthenticationError("Invalid API key")
                    
                elif response.status_code == 404:
                    logger.error(f"Resource not found: {url}")
                    raise ResourceNotFoundError(f"Resource not found: {endpoint}")
                    
                elif response.status_code >= 500:
                    # Server error, retry with backoff
                    wait_time = (2 ** attempt) + 1
                    logger.warning(f"Server error {response.status_code}. Retrying in {wait_time}s.")
                    time.sleep(wait_time)
                    continue
                    
                else:
                    # Try to get error details from response
                    try:
                        error_data = response.json()
                        error_message = error_data.get("message", f"API error: {response.status_code}")
                    except:
                        error_message = f"API error: {response.status_code}"
                    
                    logger.error(error_message)
                    raise APIError(error_message, response.status_code)
                
            except requests.exceptions.RequestException as e:
                # Network error, retry with backoff if attempts remain
                if attempt < retry_count - 1:
                    wait_time = (2 ** attempt) + 1
                    logger.warning(f"Request failed: {e}. Retrying in {wait_time}s.")
                    time.sleep(wait_time)
                else:
                    logger.error(f"Request failed after {retry_count} attempts: {e}")
                    raise ConnectionError(f"Failed to connect to API: {e}")
        
        # This should never be reached due to the exceptions above
        return None

# Custom exception classes
class AuthenticationError(Exception):
    """Raised when authentication fails"""
    pass

class ResourceNotFoundError(Exception):
    """Raised when a requested resource doesn't exist"""
    pass

class APIError(Exception):
    """Generic API error"""
    def __init__(self, message, status_code=None):
        self.status_code = status_code
        super().__init__(message)

# Example usage
if __name__ == "__main__":
    # Initialize API client
    api = ProductAPI(
        api_url="https://api.example.com/v1",
        api_key="your_api_key_here"  # Better to use environment variable
    )
    
    try:
        # Get all products
        products = api.get_products(limit=10)
        print(f"Found {len(products['data'])} products:")
        for product in products['data']:
            print(f"{product['name']} - ${product['price']} - SKU: {product['sku']}")
        
        # Search for products
        search_results = api.search_products("smartphone")
        print(f"\nSearch results for 'smartphone': {len(search_results['data'])} products found")
        
        # Create a new product
        new_product = {
            "name": "Wireless Headphones",
            "description": "Premium wireless headphones with noise cancellation",
            "price": 129.99,
            "sku": "WHEAD-101",
            "category_id": "electronics",
            "stock": 45,
            "attributes": {
                "color": "black",
                "bluetooth_version": "5.0",
                "battery_life": "20 hours"
            }
        }
        
        result = api.create_product(new_product)
        print(f"\nCreated new product: {result['name']} (ID: {result['id']})")
        
        # Update the product
        update_data = {
            "price": 119.99,
            "stock": 50,
            "attributes": {
                "on_sale": True,
                "discount_reason": "Summer Sale"
            }
        }
        
        updated = api.update_product(result['id'], update_data)
        print(f"\nUpdated product price to ${updated['price']}")
        
        # Get categories
        categories = api.get_categories()
        print("\nAvailable categories:")
        for category in categories:
            print(f"- {category['name']} ({category['id']})")
        
    except AuthenticationError as e:
        print(f"Authentication error: {e}")
    except ResourceNotFoundError as e:
        print(f"Not found: {e}")
    except APIError as e:
        print(f"API error ({e.status_code}): {e}")
    except ConnectionError as e:
        print(f"Connection error: {e}")
    except ValueError as e:
        print(f"Validation error: {e}")
```

### Example 4: Payment Processing Integration

**Requirements:**
- Process credit card payments
- Handle different payment methods
- Support refunds and payment status checks

**API Choice:** Stripe API

**Implementation in JavaScript (Node.js):**

```javascript
// payment-service.js
const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
const logger = require('./logger');

class PaymentService {
  constructor(apiKey = process.env.STRIPE_SECRET_KEY) {
    this.stripe = require('stripe')(apiKey);
  }

  /**
   * Create a payment intent for a credit card payment
   */
  async createPaymentIntent(amount, currency, customerId, description) {
    try {
      logger.info(`Creating payment intent for ${currency} ${amount / 100}`);
      
      const paymentIntent = await this.stripe.paymentIntents.create({
        amount: amount, // amount in cents
        currency: currency,
        customer: customerId,
        description: description,
        payment_method_types: ['card'],
        capture_method: 'automatic'
      });
      
      logger.info(`Created payment intent: ${paymentIntent.id}`);
      return {
        success: true,
        paymentIntentId: paymentIntent.id,
        clientSecret: paymentIntent.client_secret,
        status: paymentIntent.status
      };
    } catch (error) {
      logger.error(`Error creating payment intent: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * Create a customer in Stripe
   */
  async createCustomer(email, name, metadata = {}) {
    try {
      logger.info(`Creating customer for email: ${email}`);
      
      const customer = await this.stripe.customers.create({
        email: email,
        name: name,
        metadata: metadata
      });
      
      logger.info(`Created customer: ${customer.id}`);
      return {
        success: true,
        customerId: customer.id,
        email: customer.email
      };
    } catch (error) {
      logger.error(`Error creating customer: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * Add a payment method to a customer
   */
  async attachPaymentMethodToCustomer(customerId, paymentMethodId) {
    try {
      logger.info(`Attaching payment method ${paymentMethodId} to customer ${customerId}`);
      
      await this.stripe.paymentMethods.attach(paymentMethodId, {
        customer: customerId,
      });
      
      // Set as default payment method
      await this.stripe.customers.update(customerId, {
        invoice_settings: {
          default_payment_method: paymentMethodId,
        },
      });
      
      logger.info(`Payment method attached and set as default`);
      return {
        success: true,
        customerId: customerId,
        paymentMethodId: paymentMethodId
      };
    } catch (error) {
      logger.error(`Error attaching payment method: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * Process a payment with an existing payment method
   */
  async processPayment(amount, currency, customerId, paymentMethodId, description) {
    try {
      logger.info(`Processing payment of ${currency} ${amount / 100} with payment method ${paymentMethodId}`);
      
      const paymentIntent = await this.stripe.paymentIntents.create({
        amount: amount,
        currency: currency,
        customer: customerId,
        payment_method: paymentMethodId,
        description: description,
        confirm: true, // Confirm the payment intent immediately
        off_session: true // Customer is not present
      });
      
      logger.info(`Payment processed: ${paymentIntent.id} (${paymentIntent.status})`);
      return {
        success: true,
        paymentIntentId: paymentIntent.id,
        status: paymentIntent.status
      };
    } catch (error) {
      logger.error(`Error processing payment: ${error.message}`);
      
      // Check if payment requires authentication
      if (error.code === 'authentication_required') {
        return {
          success: false,
          requiresAuthentication: true,
          paymentIntentId: error.raw.payment_intent.id,
          clientSecret: error.raw.payment_intent.client_secret,
          error: "This payment requires authentication"
        };
      }
      
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * Issue a refund for a payment
   */
  async refundPayment(paymentIntentId, amount = null, reason = 'requested_by_customer') {
    try {
      logger.info(`Refunding payment ${paymentIntentId}`);
      
      const refundParams = {
        payment_intent: paymentIntentId,
        reason: reason
      };
      
      // If amount is specified, add it to refund only that amount
      if (amount !== null) {
        refundParams.amount = amount;
      }
      
      const refund = await this.stripe.refunds.create(refundParams);
      
      logger.info(`Refund issued: ${refund.id}`);
      return {
        success: true,
        refundId: refund.id,
        status: refund.status
      };
    } catch (error) {
      logger.error(`Error refunding payment: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * Check status of a payment intent
   */
  async checkPaymentStatus(paymentIntentId) {
    try {
      logger.info(`Checking status of payment ${paymentIntentId}`);
      
      const paymentIntent = await this.stripe.paymentIntents.retrieve(paymentIntentId);
      
      logger.info(`Payment status: ${paymentIntent.status}`);
      return {
        success: true,
        paymentIntentId: paymentIntent.id,
        status: paymentIntent.status,
        amount: paymentIntent.amount,
        currency: paymentIntent.currency,
        customerId: paymentIntent.customer,
        paymentMethodId: paymentIntent.payment_method
      };
    } catch (error) {
      logger.error(`Error checking payment status: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }

  /**
   * List payment methods for a customer
   */
  async listPaymentMethods(customerId, type = 'card') {
    try {
      logger.info(`Listing ${type} payment methods for customer ${customerId}`);
      
      const paymentMethods = await this.stripe.paymentMethods.list({
        customer: customerId,
        type: type
      });
      
      logger.info(`Found ${paymentMethods.data.length} payment methods`);
      return {
        success: true,
        paymentMethods: paymentMethods.data.map(pm => ({
          id: pm.id,
          type: pm.type,
          createdAt: new Date(pm.created * 1000),
          isDefault: pm.is_default,
          card: type === 'card' ? {
            brand: pm.card.brand,
            last4: pm.card.last4,
            expMonth: pm.card.exp_month,
            expYear: pm.card.exp_year
          } : null
        }))
      };
    } catch (error) {
      logger.error(`Error listing payment methods: ${error.message}`);
      return {
        success: false,
        error: error.message,
        code: error.code
      };
    }
  }
}

module.exports = PaymentService;

// Example usage in Express.js
const express = require('express');
const router = express.Router();
const PaymentService = require('./payment-service');
const paymentService = new PaymentService();

// Create payment intent endpoint
router.post('/create-payment-intent', async (req, res) => {
  const { amount, currency, customerId, description } = req.body;
  
  if (!amount || !currency || !customerId) {
    return res.status(400).json({ 
      success: false, 
      error: 'Missing required parameters' 
    });
  }
  
  const result = await paymentService.createPaymentIntent(
    amount, 
    currency, 
    customerId, 
    description
  );
  
  if (result.success) {
    res.json(result);
  } else {
    res.status(400).json(result);
  }
});

// Create customer endpoint
router.post('/create-customer', async (req, res) => {
  const { email, name, metadata } = req.body;
  
  if (!email || !name) {
    return res.status(400).json({ 
      success: false, 
      error: 'Email and name are required' 
    });
  }
  
  const result = await paymentService.createCustomer(email, name, metadata);
  
  if (result.success) {
    res.json(result);
  } else {
    res.status(400).json(result);
  }
});

// Process payment endpoint
router.post('/process-payment', async (req, res) => {
  const { 
    amount, 
    currency, 
    customerId, 
    paymentMethodId, 
    description 
  } = req.body;
  
  if (!amount || !currency || !customerId || !paymentMethodId) {
    return res.status(400).json({ 
      success: false, 
      error: 'Missing required parameters' 
    });
  }
  
  const result = await paymentService.processPayment(
    amount, 
    currency, 
    customerId, 
    paymentMethodId, 
    description
  );
  
  res.json(result);
});

// Issue refund endpoint
router.post('/refund', async (req, res) => {
  const { paymentIntentId, amount, reason } = req.body;
  
  if (!paymentIntentId) {
    return res.status(400).json({ 
      success: false, 
      error: 'Payment intent ID is required' 
    });
  }
  
  const result = await paymentService.refundPayment(
    paymentIntentId, 
    amount, 
    reason
  );
  
  if (result.success) {
    res.json(result);
  } else {
    res.status(400).json(result);
  }
});

module.exports = router;
```

## 13. Conclusion: Building Robust API Clients

Throughout this guide, we've explored the fundamental concepts of working with APIs and built practical examples. Let's summarize the key principles for creating robust API clients:

### 1. Follow a Layered Design

Structure your API clients with clear separation of concerns:

- **Transport Layer**: Handles HTTP requests, retries, and error handling
- **API Interface Layer**: Maps API endpoints to method calls
- **Business Logic Layer**: Transforms data for your application needs

This separation makes your code more maintainable and testable.

### 2. Implement Comprehensive Error Handling

No API is 100% reliable. Your client should:

- Catch and categorize different types of errors
- Implement appropriate retry strategies
- Provide meaningful error messages
- Fail gracefully when the API is unavailable

### 3. Be Mindful of Performance

Consider performance implications:

- Use connection pooling for multiple requests
- Implement caching where appropriate
- Batch operations when possible
- Use asynchronous requests when handling multiple calls

### 4. Make Security a Priority

Never compromise on security:

- Always use HTTPS
- Store credentials securely
- Implement proper authentication
- Validate all inputs and outputs
- Follow the principle of least privilege

### 5. Design for Testability

Make your API clients easy to test:

- Use dependency injection for external services
- Create interfaces that can be mocked
- Separate I/O operations from business logic
- Write unit tests for each component
- Use integration tests for end-to-end verification

### 6. Document Your Code

Even internal API clients need documentation:

- Document the purpose of each method
- Explain expected parameters and return values
- Provide usage examples
- Document error handling strategies
- Keep the documentation up-to-date

### 7. Be a Good API Citizen

Respect the API provider's rules:

- Follow rate limits
- Minimize unnecessary requests
- Implement exponential backoff for retries
- Keep your client libraries updated
- Review API changes and deprecation notices

### 8. Prepare for Evolution

APIs change over time, so design for adaptability:

- Version your own client code
- Create abstractions that can accommodate API changes
- Develop a strategy for handling breaking changes
- Test against API sandbox environments when available

### Final Thoughts

Building effective API clients is both an art and a science. The best implementations balance technical requirements with user needs, creating interfaces that hide complexity while providing powerful functionality.

As you develop your skills, remember that the most successful API integrations are those that users barely notice. They should work reliably, perform efficiently, and handle errors gracefully without requiring users to understand the underlying API mechanics.

By applying the principles covered in this guide, you'll be well-equipped to build API clients that stand the test of time and deliver exceptional value to your applications and users.

## 14. Further Resources

To continue your learning about APIs, here are some valuable resources:

### Books

- "RESTful Web APIs" by Leonard Richardson, Mike Amundsen, and Sam Ruby
- "Designing Web APIs" by Brenda Jin, Saurabh Sahni, and Amir Shevat
- "API Design Patterns" by JJ Geewax

### Online Courses

- "API Development in Python" (Udemy)
- "RESTful API with HTTP and JavaScript" (Coursera)
- "API Security" (Pluralsight)

### Documentation and Specifications

- [OpenAPI Specification](https://swagger.io/specification/)
- [JSON:API Specification](https://jsonapi.org/)
- [OAuth 2.0 Documentation](https://oauth.net/2/)

### Tools

- [Postman](https://www.postman.com/) - API development and testing
- [Swagger UI](https://swagger.io/tools/swagger-ui/) - API documentation
- [Charles Proxy](https://www.charlesproxy.com/) - HTTP debugging proxy

### API Directories

- [ProgrammableWeb](https://www.programmableweb.com/)
- [RapidAPI Hub](https://rapidapi.com/)
- [Public APIs](https://github.com/public-apis/public-apis)

Keep exploring, and happy coding!