## Why PIP?

"pip install" is crucial in the Python ecosystem for several reasons:


Imagine you're building a web application: You might need various libraries to 
handle web frameworks, database connections, or even data visualization. 
While Python comes with a rich standard library, many powerful tools and packages are not included by default. 
Here's where "pip install" comes into play.

### Example:

Suppose you want to build a web application using Flask, a popular web framework for Python. 
Flask isn't included in the Python standard library. To get Flask, you use "pip install":

```bash
pip install Flask
```

Without "pip install," you would have to manually download Flask and manage its dependencies, 
which can be cumbersome and error-prone. "pip install" automates this process, ensuring you 
get the correct version of Flask and all its dependencies, making your development process 
smoother and more efficient.


In essence, "pip install" allows you to easily extend Python's capabilities by accessing a 
vast repository of third-party packages (at PyPy), facilitating quicker development and 
ensuring that you can leverage the best tools available.

## So in Real Life?

Here's a simple example of a Python program using the requests package to fetch data from a web API. 
This program will make a GET request to a sample API and print the response:


Say `fetchjson.py`

```python
import requests

def get_data():
    url = "https://jsonplaceholder.typicode.com/posts/1"
    response = requests.get(url)
    
    if response.status_code == 200:
        print("Data retrieved successfully:")
        print(response.json())
    else:
        print("Failed to retrieve data. Status code:", response.status_code)

if __name__ == "__main__":
    get_data()
```

but when you run this code...

```bash
Traceback (most recent call last):
  File "<python-input-0>", line 1, in <module>
    import requests
ModuleNotFoundError: No module named 'requests'
Traceback (most recent call last):
  File "<python-input-0>", line 14, in <module>
    get_data()
    ~~~~~~~~^^
  File "<python-input-0>", line 5, in get_data
    response = requests.get(url)
               ^^^^^^^^
NameError: name 'requests' is not defined
```

Hmm.

### PIP to the rescue.

Notice the `import requests` line at the top of the code sample.

That's the error. 
Our `python` environment doesn't have a copy of `requests` and can't find the code.

_What is `requests` anyway? Well, check out [requests 2.32.3](https://pypi.org/project/requests/)_

So we need to get a copy of this package.
In a terminal, we can `pip install requests` and we see...

```bash
(venv) foo@bar ~ % pip install requests
Collecting requests
  Downloading requests-2.32.3-py3-none-any.whl.metadata (4.6 kB)
Collecting charset-normalizer<4,>=2 (from requests)
  Downloading charset_normalizer-3.4.0-cp313-cp313-macosx_11_0_arm64.whl.metadata (34 kB)
Collecting idna<4,>=2.5 (from requests)
  Downloading idna-3.10-py3-none-any.whl.metadata (10 kB)
Collecting urllib3<3,>=1.21.1 (from requests)
  Downloading urllib3-2.2.3-py3-none-any.whl.metadata (6.5 kB)
Collecting certifi>=2017.4.17 (from requests)
  Downloading certifi-2024.8.30-py3-none-any.whl.metadata (2.2 kB)
Downloading requests-2.32.3-py3-none-any.whl (64 kB)
Downloading certifi-2024.8.30-py3-none-any.whl (167 kB)
Downloading charset_normalizer-3.4.0-cp313-cp313-macosx_11_0_arm64.whl (119 kB)
Downloading idna-3.10-py3-none-any.whl (70 kB)
Downloading urllib3-2.2.3-py3-none-any.whl (126 kB)
Installing collected packages: urllib3, idna, charset-normalizer, certifi, requests
Successfully installed certifi-2024.8.30 charset-normalizer-3.4.0 idna-3.10 requests-2.32.3 urllib3-2.2.3

[notice] A new release of pip is available: 24.2 -> 24.3.1
[notice] To update, run: pip install --upgrade pip
```

And now when we run `fetchjson.py`


```bash
(venv) foo@bar ~ % python3 fetchjson.py 
Data retrieved successfully:
{'userId': 1, 'id': 1, 'title': 'sunt aut facere repellat provident occaecati excepturi optio reprehenderit',
 'body': 'quia et suscipit\nsuscipit recusandae consequuntur reprehenderit sunt rem eveniet architecto'}
```

**And you still need to to understand how to use [Python Virtual Environments](https://zcw.guru/kristofer/python3-venvs).**

